use lang:bs;
use lang:bs:macro;
use core:io;
use util:serialize;

// Check pointers for equality: if a is b, then c is d.
private Bool check(Object a, Object b, Object c, Object d) {
	if (a is b)
		c is d;
	else
		c !is d;
}


/**
 * Serializable class.
 */
class Data : serializable {
	Int a;
	Int b;
	Employee data1;
	Employee data2;
	Employee data3;

	init() {
		Employee shared;
		init() {
			a = 5;
			b = 8;
			data1 = shared;
			data2 = Manager();
			data3 = shared;
		}
	}

	void toS(StrBuf to) {
		to << "{ a: " << a << ", b: " << b << ", data1: " << data1 << ", data2: " << data2 << ", data3: " << data3 << " }";
	}

	// Write the members in another order. Should still be readable by 'read'.
	// Note: Earlier the Employee class was shuffled as well. Since we now refer to Types directly,
	// we can no longer do that.
	void writeShuffled(ObjOStream to) {
		SerializedStdType desc(named{Data}, &__init(Data, ObjIStream));
		desc.add("b", named{core:Int});
		desc.add("data2", named{Employee});
		desc.add("data3", named{Employee});
		desc.add("a", named{core:Int});
		desc.add("data1", named{Employee});

		if (to.startClass(desc, this)) {
			b.write(to);
			data2.write(to);
			data3.write(to);
			a.write(to);
			data1.write(to);
			to.end();
		}
	}

	// Compare to another object to see if their structure are the same.
	Bool compare(Data other) {
		Bool ok = true;
		ok &= a == other.a;
		ok &= b == other.b;
		ok &= data1.compare(other.data1);
		ok &= data2.compare(other.data2);
		ok &= data3.compare(other.data3);

		// Check so that references are properly preserved.
		ok &= check(data1, data2, other.data1, other.data2);
		ok &= check(data1, data3, other.data1, other.data3);
		ok &= check(data2, data3, other.data2, other.data3);

		ok;
	}
}

/**
 * Serializable class containing an array.
 */
class ArrayData : serializable {
	Int a;
	Int[] numbers;
	Employee[] data;

	init() {
		Employee e;
		Manager m;
		init { a = 20; }
		numbers << 1 << 3 << 5;
		data << e << m << e << m << Employee();
	}

	void toS(StrBuf to) {
		to << "{ a: " << a << ", numbers: " << numbers << ", data: " << data << " }";
	}

	Bool compare(ArrayData other) {
		Bool ok = a == other.a;
		if (data.count != other.data.count)
			return false;

		for (i, a in numbers) {
			ok &= a == other.numbers[i];
		}

		for (i, a in data) {
			for (j, b in data) {
				ok &= check(a, b, other.data[i], other.data[j]);
			}

			ok &= a.compare(other.data[i]);
		}

		ok;
	}
}

/**
 * Serialization of various Maybe-types.
 */
class MaybeData : serializable {
	Employee? a;
	Employee? b;

	Int? c;
	Int? d;

	Salary? e;
	Salary? f;

	init() {
		init {
			a = Employee();
			c = 20;
			e = Salary(128, "USD");
		}
	}

	void toS(StrBuf to) {
		// TODO: Why don't << work out of the box?
		to << "{a: ${a}, b: ${b}, c: ${c}, d: ${d}, e: ${e}, f: ${f} }";
	}

	Bool compare(MaybeData other) {
		Bool ok = true;
		ok &= a.any == other.a.any;
		ok &= b.any == other.b.any;
		ok &= c.any == other.c.any;
		ok &= d.any == other.d.any;
		ok &= e.any == other.e.any;
		ok &= f.any == other.f.any;

		// Crude, but OK.
		ok &= toS == other.toS;
		ok;
	}
}


/**
 * A serializable value.
 */
value Salary : serializable {
	Nat amount;
	Str currency;

	init(Nat amount, Str currency) {
		init {
			amount = amount;
			currency = currency;
		}
	}

	void writeShuffled(ObjOStream to) {
		SerializedStdType desc(named{Salary}, &__init(Salary, ObjIStream));
		desc.add("currency", named{Str});
		desc.add("amount", named{Nat});
		if (to.startValue(desc)) {
			currency.write(to);
			amount.write(to);
			to.end();
		}
	}

	Bool ==(Salary other) {
		amount == other.amount & currency == other.currency;
	}

	void toS(StrBuf to) {
		to << amount << " " << currency;
	}
}


/**
 * Another serializable class.
 */
class Employee : serializable {
	Str name;
	Int coolness;
	Salary salary;

	init() {
		init {
			name = "John Smith";
			coolness = 500;
			salary = Salary(100, "SEK");
		}
	}

	Bool compare(Employee other) {
		if (!sameType(this, other))
			return false;

		Bool ok = name == other.name;
		ok &= coolness == other.coolness;
		ok &= salary == other.salary;

		ok;
	}

	void toS(StrBuf to) {
		to << "Employee: { name: " << name << ", coolness: " << coolness << ", salary: " << salary << " }";
	}
}


/**
 * A sub-class to Employee.
 */
class Manager : extends Employee, serializable {
	Str title;

	init() {
		init() { title = "Executive"; }
		name = "Mr. " # name;
		coolness *= 10;
		salary.amount *= 10;
	}

	Bool compare(Employee other) {
		if (other as Manager) {
			Bool ok = super:compare(other);
			ok &= title == other.title;
			ok;
		} else {
			false;
		}
	}

	void toS(StrBuf to) {
		to << "Manager: { name: " << name << ", coolness: " << coolness << ", salary: " << salary << ", title: " << title << " }";
	}
}

/**
 * A class with completely custom serialization.
 */
class CustomData : serializable {
	Int a;
	Int b;

	init() {
		init {
			a = 20;
			b = 30;
		}
	}

	init(ObjIStream from) {
		init {
			a = Byte:read(from.from).int;
			b = Byte:read(from.from).int;
		}
		from.end();
	}

	void write(ObjOStream to) {
		if (to.startClass(named{CustomData}, this)) {
			a.byte.write(to.to);
			b.byte.write(to.to);
			to.end();
		}
	}

	Bool compare(CustomData other) {
		return a == other.a & b == other.b;
	}
}


// Show the string representation of serialized data.
private void showString(Buffer data) {
	print(TextObjStream(MemIStream(data)).read());
}

// Just serialize an object and then de-serialize it.
Bool simpleSerialization() {
	Data original;

	MemOStream out;
	original.write(ObjOStream(out));

	// showString(out.buffer);

	MemIStream in(out.buffer);
	Data copy = Data:read(ObjIStream(in));

	original.compare(copy);
}

// Serialize one object multiple times in the same stream.
Bool multipleSerialization() {
	Data original;

	MemOStream out;
	ObjOStream objOut(out);
	original.write(objOut);
	Nat size1 = out.buffer.filled;
	original.write(objOut);
	Nat size2 = out.buffer.filled - size1;

	// The second serialization should be smaller, as it does not require duplicating metadata.
	if (size2 >= size1) {
		print("Second serialization should be smaller!");
		return false;
	}

	ObjIStream objIn(MemIStream(out.buffer));
	Data copy1 = Data:read(objIn);
	Data copy2 = Data:read(objIn);

	// These are separate entities, and should not have anything in common, as if we would have
	// cloned them.
	if (copy1 is copy2) {
		print("The objects are the same!");
		return false;
	}
	if (copy1.data1 is copy2.data1) {
		print("The objects share data!");
		return false;
	}

	return original.compare(copy1) & original.compare(copy2);
}

// Serialize an object, but with some members in the wrong order!
Bool shuffledSerialization() {
	Data original;

	MemOStream out;
	original.writeShuffled(ObjOStream(out));

	// showString(out.buffer);

	MemIStream in(out.buffer);
	Data copy = Data:read(ObjIStream(in));

	original.compare(copy);
}

/**
 * Serializable class, with the ability to write a subset, and a superset of members so that we can
 * test our ability to create new values, and account for extra data.
 */
class MissingData : serializable {
	Str a = "Def A";
	Str b = "Def B";

	init(Str a, Str b) {
		init {
			a = a;
			b = b;
		}
	}

	void writeShuffled(ObjOStream to) {
		SerializedStdType desc(named{MissingData}, &__init(MissingData, ObjIStream));
		desc.add("b", named{core:Str});
		desc.add("a", named{core:Str});

		if (to.startClass(desc, this)) {
			b.write(to);
			a.write(to);
			to.end();
		}
	}

	void writeExtra(ObjOStream to) {
		SerializedStdType desc(named{MissingData}, &__init(MissingData, ObjIStream));
		desc.add("extra1", named{core:Str});
		desc.add("a", named{core:Str});
		desc.add("extra2", named{core:Int});
		desc.add("b", named{core:Str});
		desc.add("extra3", named{core:Str});

		if (to.startClass(desc, this)) {
			"extra1".write(to);
			a.write(to);
			10.write(to); // as "extra2"
			b.write(to);
			"extra3".write(to);
			to.end();
		}
	}

	void writeMissing(ObjOStream to) {
		SerializedStdType desc(named{MissingData}, &__init(MissingData, ObjIStream));
		desc.add("b", named{core:Str});

		if (to.startClass(desc, this)) {
			b.write(to);
			to.end();
		}
	}

	void toS(StrBuf to) {
		to << a << ", " << b;
	}
}


// Serialize an object, but with some new members (that have default values).
Str shuffledSerialization2() {
	MissingData original("Custom A", "Custom B");
	MemOStream out;
	original.writeShuffled(ObjOStream(out));

	// showString(out.buffer);

	MemIStream in(out.buffer);
	MissingData copy = MissingData:read(ObjIStream(in));

	copy.toS;
}

// Serialize an object, but have the stream contain extra members.
Str extraSerialization() {
	MissingData original("Custom A", "Custom B");
	MemOStream out;
	original.writeExtra(ObjOStream(out));

	// showString(out.buffer);

	MemIStream in(out.buffer);
	MissingData copy = MissingData:read(ObjIStream(in));

	copy.toS;
}

// Serialize an object, but with some new members (that have default values).
Str missingSerialization() {
	MissingData original("Custom A", "Custom B");
	MemOStream out;
	original.writeMissing(ObjOStream(out));

	// showString(out.buffer);

	MemIStream in(out.buffer);
	MissingData copy = MissingData:read(ObjIStream(in));

	copy.toS;
}

// Serialize some primitives.
Bool primitiveSerialization() {
	MemOStream out;
	ObjOStream objOut(out);
	1b.write(objOut);
	1l.write(objOut);
	"Hello".write(objOut);

	ObjIStream objIn(MemIStream(out.buffer));
	Bool ok = true;
	ok &= Byte:read(objIn) == 1b;
	ok &= Long:read(objIn) == 1l;
	ok &= Str:read(objIn) == "Hello";
	ok;
}

// Read and write objects with differing formal types at top-level.
Bool typeDiffSerialization() {
	Manager original;

	MemOStream out;
	ObjOStream objOut(out);
	original.write(objOut);

	ObjIStream objIn(MemIStream(out.buffer));
	var e = Employee:read(objIn);

	original.compare(e);
}

// Array serialization.
Bool arraySerialization() {
	ArrayData data;
	MemOStream out;
	ObjOStream objOut(out);
	data.write(objOut);

	// showString(out.buffer);

	ObjIStream objIn(MemIStream(out.buffer));
	var copy = ArrayData:read(objIn);

	data.compare(copy);
}

// Map serialization.
Bool mapSerialization() {
	Int->Employee data;
	Employee e;
	data.put(1, Employee());
	data.put(5, e);
	data.put(8, e);
	data.put(3, Manager());

	MemOStream out;
	ObjOStream objOut(out);
	data.write(objOut);

	// showString(out.buffer);

	ObjIStream objIn(MemIStream(out.buffer));
	var copy = Map<Int, Employee>:read(objIn);

	if (copy.count != data.count)
		return false;

	Bool ok = true;
	for (k1, v1 in data) {
		for (k2, v2 in data) {
			ok &= check(v1, v2, copy.get(k1), copy.get(k2));
		}

		ok &= v1.compare(copy.get(k1));
	}
	ok;
}

// Set serialization.
Bool setSerialization() {
	Set<Str> data;
	data.put("Hello");
	data.put("Goodbye");
	data.put("Storm");

	Set<Int> ints;
	ints.put(5);
	ints.put(7);
	ints.put(18);

	MemOStream out;
	ObjOStream objOut(out);
	data.write(objOut);
	ints.write(objOut);

	ObjIStream objIn(MemIStream(out.buffer));
	var copy = Set<Str>:read(objIn);
	var intsCopy = Set<Int>:read(objIn);

	Bool ok = true;
	for (k in data)
		ok &= copy.has(k);
	ok &= data.count == copy.count;

	for (k in ints)
		ok &= intsCopy.has(k);
	ok &= ints.count == intsCopy.count;

	ok;
}

// Maybe serialization.
Bool maybeSerialization() {
	MaybeData data;
	MemOStream out;
	ObjOStream objOut(out);
	data.write(objOut);

	// showString(out.buffer);

	ObjIStream objIn(MemIStream(out.buffer));
	var copy = MaybeData:read(objIn);

	data.compare(copy);
}

// Custom serialization.
Bool customSerialization() {
	CustomData data;
	MemOStream out;
	ObjOStream objOut(out);
	data.write(objOut);

	// showString(out.buffer);

	ObjIStream objIn(MemIStream(out.buffer));
	var copy = CustomData:read(objIn);

	data.compare(copy);
}

// URL serialization.
Bool urlSerialization() {
	Url url = cwdUrl() / "test.bs";

	MemOStream out;
	ObjOStream objOut(out);
	url.write(objOut);

	// showString(out.buffer);

	ObjIStream objIn(MemIStream(out.buffer));
	var copy = Url:read(objIn);

	copy.toS == url.toS;
}

// Version serialization.
Bool versionSerialization() {
	unless (version = core:info:parseVersion("1.0.2-test+abc"))
		return false;

	MemOStream out;
	ObjOStream objOut(out);
	version.write(objOut);

	// showString(out.buffer);

	ObjIStream objIn(MemIStream(out.buffer));
	var copy = core:info:Version:read(objIn);

	copy.toS == version.toS;
}

// Create a fairly large serialized blob. Used to test read limits in C++ later on.
Buffer createLargeObject() {
	ArrayData data;
	for (Nat i = 0; i < 100; i++) {
		data.numbers << (i + 5).int;
		data.data << Employee();
	}

	MemOStream out;
	ObjOStream objOut(out);
	data.write(objOut);

	out.buffer;
}

core:lang:Type largeObjectType() {
	named{ArrayData};
}


// Deserialize some data in a buffer (from C++). This is used to ensure that we don't crash in cases
// we previously found to crash the deserialization logic. While we don't expect random data to make
// sense, we don't expect it to crash the system.
void deserializeBuffer(Buffer data) {
	ObjIStream is(MemIStream(data));
	try {
		Data:read(is);
	} catch (Exception e) {
		// This is fine, and often expected for these tests.
	}

	// Check that our memory is not all corrupt by calling the GC.
	gc();
}

// Deserialize random data, make sure we don't crash.
void deserializeFuzz() {
	for (Nat i = 0; i < 100; i++) {
		var data = crypto:RandomStream.read(128);
		print(data.toS);

		ObjIStream obj(MemIStream(data));
		obj.maxSize = 32768; // Limit to 32KiB to avoid performance degradation.
		try {
			Data:read(obj);
			print("Read OK!");
		} catch (Exception e) {
			print("Error: ${e.toS}");
		}

		// Basic sanity checking by calling the GC.
		gc();
	}
}

// Simple measurements of serialization performance.
void serializationPerf() {
	Data original;
	Nat runs = 1000;
	Nat times = 1000;

	MemOStream out;
	Moment start1;
	for (Nat i = 0; i < runs; i++) {
		out = MemOStream();
		ObjOStream objOut(out);
		for (Nat j = 0; j < times; j++) {
			original.write(objOut);
		}
	}
	Moment end1;

	Moment start2;
	for (Nat i = 0; i < runs; i++) {
		ObjIStream objIn(MemIStream(out.buffer));
		for (Nat j = 0; j < times; j++) {
			Data:read(objIn);
		}
	}
	Moment end2;

	print("Serialization time for ${runs} runs, ${times} times:");
	print("Serialized size: ${out.buffer.filled} bytes");
	print("Serialization  : ${end1 - start1}");
	print("Deserialization: ${end2 - start1}");
}
