/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

/**
* Read the entire contents of a WebAssembly file.
*
* @module @stdlib/fs/read-wasm
*
* @example
* var join = require( 'path' ).join;
* var readWASM = require( '@stdlib/fs/read-wasm' );
*
* var fpath = join( __dirname, 'foo.wasm' );
* readWASM( fpath, onRead );
*
* function onRead( error, buf ) {
*     if ( error ) {
*         throw error;
*     }
*     console.log( out );
* }
*
* @example
* var join = require( 'path' ).join;
* var instanceOf = require( '@stdlib/assert/instance-of' );
* var readWASMSync = require( '@stdlib/fs/read-wasm' ).sync;
*
* var fpath = join( __dirname, 'foo.wasm' );
* var out = readWASMSync( fpath );
* if ( instanceOf( out, Error ) ) {
*     throw out;
* }
* console.log( out );
*/

// MODULES //

var setReadOnly = require( '@stdlib/utils/define-nonenumerable-read-only-property' );
var readWASM = require( './async.js' );
var sync = require( './sync.js' );


// MAIN //

setReadOnly( readWASM, 'sync', sync );


// EXPORTS //

module.exports = readWASM;
