//***************************************************************************
//
//  Copyright (c) 1999 - 2006 Intel Corporation
//
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
//
//***************************************************************************

/**
	@file IFXErrorInfo.h

	This header file defines the IFXErrorInfo interface and its functionality.
*/


#ifndef __IFXErrorInfo_H__
#define __IFXErrorInfo_H__


#include "IFXUnknown.h"
#include "IFXTask.h"
#include "IFXSchedulerTypes.h"


//-------------------------------------------------------------------
// Interface
//-------------------------------------------------------------------

/**
	This is the main interface for IFXErrorInfo. When an IFXTask::Execute
	method produces an error, an error notification is generated by 
	submitting information about the error, represented by a component that 
	supports this interface. The error notification can be handled by an 
	IFXTask handler registered with the Notification Manager; the handler 
	should be registered as IFXNotificationType_Error and the 
	IFXNotificationId should be equal to the IFXTaskHandle of the registered 
	task that will be monitored. To monitor all registered tasks, use 
	IFXNotificationId_Nil. The handler will receive a reference to an 
	IFXNotificationInfo interface (supported by a simple data holding 
	component) as the argument to its Execute method. If no IFXTask is 
	registered to handle errors, the error is discarded.

	@note	The associated IID is named IID_IFXErrorInfo, and the CID for 
			this component is CID_IFXErrorInfo.
*/
class IFXErrorInfo : public IFXTaskData
{
public:
	/**
		This method retrieves an IFXTask that generated the error.

		@param  ppTask  Returns the IFXTask pointer.\n

		@return Returns an IFXRESULT code..\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the IFXErrorInfo object has not been initialized.\n
		- @b IFX_E_INVALID_POINTER is returned if ppTask is invalid.\n
	*/
	virtual IFXRESULT IFXAPI  GetErrorTask(IFXTask ** ppTask) = 0;
	/**
		This method sets the IFXTask that generated the error.

		@param  pTask The IFXTask pointer. This pointer may be NULL.\n

		@return Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the IFXErrorInfo object has not been initialized.\n
	*/
	virtual IFXRESULT IFXAPI  SetErrorTask(IFXTask * pTask) = 0;
	/**
		This method retrieves the IFXTaskHandle that uniquely identifies the registered IFXTask that generated the error.

		@param  pTaskHandle Returns the task handle.\n

		@return Returns an IFXRESULT code..\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the IFXErrorInfo object has not been initialized.\n
		- @b IFX_E_INVALID_POINTER is returned if pTaskHandle is invalid.\n
	*/
	virtual IFXRESULT IFXAPI  GetErrorTaskHandle(IFXTaskHandle *pTaskHandle) = 0;
	/**
		This method sets the IFXTaskHandle that uniquely identifies the registered IFXTask that generated the error.

		@param  taskHandle  The task handle.\n

		@return Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the IFXErrorInfo object has not been initialized.\n
	*/
	virtual IFXRESULT IFXAPI  SetErrorTaskHandle(IFXTaskHandle taskHandle) = 0;
	/**
		This method retrieves the error code that identifies the type of error.

		@param  pErrorCode  Returns the IFXRESULT error code.\n

		@return Returns an IFXRESULT code..\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the IFXErrorInfo object has not been initialized.\n
		- @b IFX_E_INVALID_POINTER is returned if pErrorCode is invalid.\n
	*/
	virtual IFXRESULT IFXAPI  GetErrorCode(IFXRESULT *pErrorCode) = 0;
	/**
		This method sets the error code that identifies the type of error..

		@param  errorCode The IFXRESULT error code.\n

		@return Returns an IFXRESULT code.\n
		- @b IFX_OK upon success.\n
		- @b IFX_E_NOT_INITIALIZED if the IFXErrorInfo object has not been initialized.\n
	*/
	virtual IFXRESULT IFXAPI  SetErrorCode(IFXRESULT errorCode) = 0;
};


// {76E95C84-A633-11d5-9AE3-00D0B73FB755}
IFXDEFINE_GUID(IID_IFXErrorInfo,
			   0x76e95c84, 0xa633, 0x11d5, 0x9a, 0xe3, 0x0, 0xd0, 0xb7, 0x3f, 0xb7, 0x55);


#endif


