/*
 * Copyright (C) 2012 Michael Brown <mbrown@fensystems.co.uk>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 * You can also choose to distribute this program under the terms of
 * the Unmodified Binary Distribution Licence (as given in the file
 * COPYING.UBDL), provided that you have satisfied its requirements.
 */

FILE_LICENCE ( GPL2_OR_LATER_OR_UBDL );

/** @file
 *
 * OCSP tests
 *
 *
 * Test vectors generated using OpenSSL:
 *
 *     openssl ocsp -no_nonce -issuer issuer.crt -cert cert.crt \
 *		    -url http://ocsp.server.address \
 *		    -reqout request.der -respout response.der
 */

/* Forcibly enable assertions */
#undef NDEBUG

#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <ipxe/x509.h>
#include <ipxe/rootcert.h>
#include <ipxe/ocsp.h>
#include <ipxe/test.h>

/** An OCSP test certificate */
struct ocsp_test_certificate {
	/** Data */
	const void *data;
	/** Length of data */
	size_t len;
	/** Parsed certificate */
	struct x509_certificate *cert;
};

/** An OCSP test */
struct ocsp_test {
	/** Certificate */
	struct ocsp_test_certificate *cert;
	/** Issuing certificate */
	struct ocsp_test_certificate *issuer;
	/** Request */
	const void *request;
	/** Length of request */
	size_t request_len;
	/** Response */
	const void *response;
	/** Length of response */
	size_t response_len;
	/* OCSP check */
	struct ocsp_check *ocsp;
};

/** Define inline data */
#define DATA(...) { __VA_ARGS__ }

/** Define a test certificate */
#define CERTIFICATE( name, DATA, FINGERPRINT )				\
	static const uint8_t name ## _data[] = DATA;			\
	static struct ocsp_test_certificate name = {			\
		.data = name ## _data,					\
		.len = sizeof ( name ## _data ),			\
	}

/** Define an OCSP test */
#define OCSP( name, CERT, ISSUER, REQUEST, RESPONSE )			\
	static const uint8_t name ## _request[] = REQUEST;		\
	static const uint8_t name ## _response[] = RESPONSE;		\
	static struct ocsp_test name = {				\
		.cert = CERT,						\
		.issuer = ISSUER,					\
		.request = name ## _request,				\
		.request_len = sizeof ( name ## _request ),		\
		.response = name ## _response,				\
		.response_len = sizeof ( name ## _response ),		\
	 }

/**
 * Prepare an OCSP test
 *
 * @v test		OCSP test
 */
static void ocsp_prepare_test ( struct ocsp_test *test ) {
	struct x509_certificate *cert = test->cert->cert;
	struct x509_certificate *issuer = test->issuer->cert;

	/* Invalidate certificate being checked */
	x509_invalidate ( cert );

	/* Force-validate issuer certificate */
	issuer->root = &root_certificates;
	issuer->path_remaining = ( issuer->extensions.basic.path_len + 1 );
}

/*
 * subject	bank.barclays.co.uk
 * issuer	VeriSign Class 3 International Server CA - G3
 */
CERTIFICATE ( barclays_crt,
	DATA ( 0x30, 0x82, 0x05, 0x7b, 0x30, 0x82, 0x04, 0x63, 0xa0, 0x03,
	       0x02, 0x01, 0x02, 0x02, 0x10, 0x49, 0x83, 0xfc, 0x05, 0x76,
	       0xdf, 0x36, 0x91, 0x7c, 0x64, 0x2a, 0x27, 0xc1, 0xf1, 0x48,
	       0xe3, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
	       0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x81, 0xbc, 0x31,
	       0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
	       0x55, 0x53, 0x31, 0x17, 0x30, 0x15, 0x06, 0x03, 0x55, 0x04,
	       0x0a, 0x13, 0x0e, 0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67,
	       0x6e, 0x2c, 0x20, 0x49, 0x6e, 0x63, 0x2e, 0x31, 0x1f, 0x30,
	       0x1d, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x13, 0x16, 0x56, 0x65,
	       0x72, 0x69, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x54, 0x72, 0x75,
	       0x73, 0x74, 0x20, 0x4e, 0x65, 0x74, 0x77, 0x6f, 0x72, 0x6b,
	       0x31, 0x3b, 0x30, 0x39, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x13,
	       0x32, 0x54, 0x65, 0x72, 0x6d, 0x73, 0x20, 0x6f, 0x66, 0x20,
	       0x75, 0x73, 0x65, 0x20, 0x61, 0x74, 0x20, 0x68, 0x74, 0x74,
	       0x70, 0x73, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x76,
	       0x65, 0x72, 0x69, 0x73, 0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f,
	       0x6d, 0x2f, 0x72, 0x70, 0x61, 0x20, 0x28, 0x63, 0x29, 0x31,
	       0x30, 0x31, 0x36, 0x30, 0x34, 0x06, 0x03, 0x55, 0x04, 0x03,
	       0x13, 0x2d, 0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67, 0x6e,
	       0x20, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x20, 0x33, 0x20, 0x49,
	       0x6e, 0x74, 0x65, 0x72, 0x6e, 0x61, 0x74, 0x69, 0x6f, 0x6e,
	       0x61, 0x6c, 0x20, 0x53, 0x65, 0x72, 0x76, 0x65, 0x72, 0x20,
	       0x43, 0x41, 0x20, 0x2d, 0x20, 0x47, 0x33, 0x30, 0x1e, 0x17,
	       0x0d, 0x31, 0x31, 0x31, 0x32, 0x30, 0x35, 0x30, 0x30, 0x30,
	       0x30, 0x30, 0x30, 0x5a, 0x17, 0x0d, 0x31, 0x32, 0x31, 0x32,
	       0x30, 0x34, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30,
	       0x7f, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
	       0x13, 0x02, 0x47, 0x42, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03,
	       0x55, 0x04, 0x08, 0x13, 0x07, 0x45, 0x6e, 0x67, 0x6c, 0x61,
	       0x6e, 0x64, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04,
	       0x07, 0x14, 0x0a, 0x47, 0x6c, 0x6f, 0x75, 0x63, 0x65, 0x73,
	       0x74, 0x65, 0x72, 0x31, 0x1a, 0x30, 0x18, 0x06, 0x03, 0x55,
	       0x04, 0x0a, 0x14, 0x11, 0x42, 0x61, 0x72, 0x63, 0x6c, 0x61,
	       0x79, 0x73, 0x20, 0x42, 0x61, 0x6e, 0x6b, 0x20, 0x50, 0x6c,
	       0x63, 0x31, 0x0f, 0x30, 0x0d, 0x06, 0x03, 0x55, 0x04, 0x0b,
	       0x14, 0x06, 0x47, 0x4c, 0x4f, 0x2d, 0x4c, 0x32, 0x31, 0x1c,
	       0x30, 0x1a, 0x06, 0x03, 0x55, 0x04, 0x03, 0x14, 0x13, 0x62,
	       0x61, 0x6e, 0x6b, 0x2e, 0x62, 0x61, 0x72, 0x63, 0x6c, 0x61,
	       0x79, 0x73, 0x2e, 0x63, 0x6f, 0x2e, 0x75, 0x6b, 0x30, 0x82,
	       0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
	       0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01,
	       0x0f, 0x00, 0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01,
	       0x00, 0x99, 0x4c, 0x2e, 0x00, 0xa0, 0xaf, 0xe2, 0xbc, 0x52,
	       0x43, 0x83, 0x34, 0x03, 0x58, 0xdf, 0xd3, 0xea, 0x43, 0xa2,
	       0xfd, 0x2c, 0x4c, 0x3c, 0x32, 0x9c, 0x60, 0x40, 0xe5, 0xa1,
	       0x07, 0x8d, 0x32, 0x21, 0xc1, 0xbd, 0xf1, 0x04, 0x2e, 0x90,
	       0xf3, 0x05, 0x30, 0xd4, 0x6f, 0x81, 0x3e, 0x08, 0xb6, 0xc3,
	       0xc1, 0xcf, 0xc4, 0x59, 0x7c, 0x56, 0x27, 0xea, 0x74, 0xe7,
	       0x8f, 0x50, 0xd1, 0xa9, 0x13, 0x57, 0x3a, 0x05, 0x5a, 0xd7,
	       0x7f, 0xfc, 0xc5, 0xc6, 0x66, 0xec, 0xa4, 0x03, 0xec, 0x97,
	       0x1a, 0x4b, 0x28, 0xf9, 0xc9, 0xf4, 0xea, 0xc6, 0x89, 0x68,
	       0xc1, 0x42, 0xcd, 0x80, 0xfc, 0xeb, 0x86, 0x6d, 0x1c, 0xd6,
	       0xa1, 0x05, 0x16, 0xa4, 0xcf, 0x82, 0x1d, 0x07, 0x67, 0x7c,
	       0xeb, 0xa1, 0x69, 0xf3, 0xf2, 0x21, 0xa7, 0x79, 0xf5, 0xf2,
	       0xdc, 0xb6, 0x0b, 0x6e, 0x19, 0xcc, 0x50, 0x53, 0xf3, 0xbd,
	       0xb9, 0x71, 0xdc, 0x2b, 0x15, 0x78, 0x6d, 0xd4, 0xd3, 0x82,
	       0xce, 0x37, 0x0c, 0xb5, 0x5e, 0x24, 0x8d, 0x80, 0x40, 0x71,
	       0x4e, 0x7b, 0x0a, 0x6e, 0x30, 0x4b, 0xb6, 0x2c, 0x23, 0x9e,
	       0xd5, 0x08, 0x7d, 0x8a, 0x72, 0x46, 0xf6, 0x52, 0x98, 0xcb,
	       0x03, 0x79, 0x61, 0xfe, 0xc1, 0x97, 0x15, 0x4b, 0x05, 0x36,
	       0x0c, 0x11, 0xe9, 0x95, 0x4b, 0xef, 0xf3, 0x2d, 0xf3, 0xef,
	       0x33, 0x6c, 0xc6, 0x98, 0xb9, 0x65, 0xe3, 0x3c, 0x26, 0x86,
	       0xb5, 0x87, 0x9e, 0x20, 0x92, 0x7b, 0x8f, 0x13, 0x66, 0x5e,
	       0x26, 0x09, 0xd6, 0x83, 0xee, 0x56, 0x72, 0x08, 0x6c, 0x2a,
	       0x4c, 0xf2, 0x5b, 0xf1, 0x08, 0x4b, 0x91, 0x9e, 0x67, 0x37,
	       0x2f, 0xc5, 0xcf, 0x1a, 0xa8, 0xa1, 0x1c, 0xb6, 0x2d, 0xd0,
	       0x81, 0xf4, 0xf7, 0x37, 0xb9, 0xa3, 0x37, 0x3f, 0x6b, 0x2f,
	       0x62, 0x82, 0xa7, 0x17, 0xc1, 0x8c, 0x69, 0x02, 0x03, 0x01,
	       0x00, 0x01, 0xa3, 0x82, 0x01, 0xb3, 0x30, 0x82, 0x01, 0xaf,
	       0x30, 0x09, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x02, 0x30,
	       0x00, 0x30, 0x0b, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x04, 0x04,
	       0x03, 0x02, 0x05, 0xa0, 0x30, 0x41, 0x06, 0x03, 0x55, 0x1d,
	       0x1f, 0x04, 0x3a, 0x30, 0x38, 0x30, 0x36, 0xa0, 0x34, 0xa0,
	       0x32, 0x86, 0x30, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f,
	       0x53, 0x56, 0x52, 0x49, 0x6e, 0x74, 0x6c, 0x2d, 0x47, 0x33,
	       0x2d, 0x63, 0x72, 0x6c, 0x2e, 0x76, 0x65, 0x72, 0x69, 0x73,
	       0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x53, 0x56,
	       0x52, 0x49, 0x6e, 0x74, 0x6c, 0x47, 0x33, 0x2e, 0x63, 0x72,
	       0x6c, 0x30, 0x44, 0x06, 0x03, 0x55, 0x1d, 0x20, 0x04, 0x3d,
	       0x30, 0x3b, 0x30, 0x39, 0x06, 0x0b, 0x60, 0x86, 0x48, 0x01,
	       0x86, 0xf8, 0x45, 0x01, 0x07, 0x17, 0x03, 0x30, 0x2a, 0x30,
	       0x28, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x02,
	       0x01, 0x16, 0x1c, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3a, 0x2f,
	       0x2f, 0x77, 0x77, 0x77, 0x2e, 0x76, 0x65, 0x72, 0x69, 0x73,
	       0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x72, 0x70,
	       0x61, 0x30, 0x28, 0x06, 0x03, 0x55, 0x1d, 0x25, 0x04, 0x21,
	       0x30, 0x1f, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x86, 0xf8,
	       0x42, 0x04, 0x01, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05,
	       0x07, 0x03, 0x01, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05,
	       0x07, 0x03, 0x02, 0x30, 0x72, 0x06, 0x08, 0x2b, 0x06, 0x01,
	       0x05, 0x05, 0x07, 0x01, 0x01, 0x04, 0x66, 0x30, 0x64, 0x30,
	       0x24, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30,
	       0x01, 0x86, 0x18, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f,
	       0x6f, 0x63, 0x73, 0x70, 0x2e, 0x76, 0x65, 0x72, 0x69, 0x73,
	       0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x3c, 0x06,
	       0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x02, 0x86,
	       0x30, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x53, 0x56,
	       0x52, 0x49, 0x6e, 0x74, 0x6c, 0x2d, 0x47, 0x33, 0x2d, 0x61,
	       0x69, 0x61, 0x2e, 0x76, 0x65, 0x72, 0x69, 0x73, 0x69, 0x67,
	       0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x53, 0x56, 0x52, 0x49,
	       0x6e, 0x74, 0x6c, 0x47, 0x33, 0x2e, 0x63, 0x65, 0x72, 0x30,
	       0x6e, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x01,
	       0x0c, 0x04, 0x62, 0x30, 0x60, 0xa1, 0x5e, 0xa0, 0x5c, 0x30,
	       0x5a, 0x30, 0x58, 0x30, 0x56, 0x16, 0x09, 0x69, 0x6d, 0x61,
	       0x67, 0x65, 0x2f, 0x67, 0x69, 0x66, 0x30, 0x21, 0x30, 0x1f,
	       0x30, 0x07, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x04,
	       0x14, 0x4b, 0x6b, 0xb9, 0x28, 0x96, 0x06, 0x0c, 0xbb, 0xd0,
	       0x52, 0x38, 0x9b, 0x29, 0xac, 0x4b, 0x07, 0x8b, 0x21, 0x05,
	       0x18, 0x30, 0x26, 0x16, 0x24, 0x68, 0x74, 0x74, 0x70, 0x3a,
	       0x2f, 0x2f, 0x6c, 0x6f, 0x67, 0x6f, 0x2e, 0x76, 0x65, 0x72,
	       0x69, 0x73, 0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f,
	       0x76, 0x73, 0x6c, 0x6f, 0x67, 0x6f, 0x31, 0x2e, 0x67, 0x69,
	       0x66, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
	       0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01,
	       0x00, 0x49, 0xf3, 0x7c, 0x15, 0xb0, 0x50, 0x97, 0xb7, 0xcd,
	       0x87, 0x75, 0x85, 0xcc, 0x55, 0x7c, 0x62, 0x97, 0x97, 0x04,
	       0xbd, 0xc2, 0x22, 0xfc, 0xf5, 0x2c, 0x75, 0xdc, 0x25, 0x6d,
	       0xed, 0xcd, 0x22, 0x2e, 0xa4, 0xcd, 0x88, 0x95, 0xe8, 0x52,
	       0x45, 0x7a, 0xa0, 0x85, 0xcc, 0x6d, 0x1c, 0xcb, 0xd8, 0xc3,
	       0x26, 0x2c, 0xee, 0xb5, 0xe1, 0x38, 0x3f, 0xb6, 0x96, 0x10,
	       0xa3, 0xb3, 0x1e, 0x2d, 0xdc, 0xe8, 0x07, 0x2f, 0xc2, 0xb7,
	       0x50, 0xd5, 0x60, 0x73, 0x0d, 0x43, 0xb1, 0xaf, 0xd9, 0xcb,
	       0x39, 0x00, 0xc7, 0x00, 0xb3, 0x1f, 0xa4, 0xaf, 0xf8, 0xed,
	       0x9b, 0x6a, 0x7a, 0x10, 0xcc, 0x81, 0x92, 0xc2, 0x58, 0x7e,
	       0x42, 0xe4, 0xbf, 0xcf, 0x8c, 0x91, 0x7f, 0xde, 0xe6, 0xac,
	       0x37, 0x31, 0x58, 0x90, 0xee, 0x51, 0xf5, 0x0e, 0xe5, 0x84,
	       0xd8, 0x51, 0x89, 0x50, 0xfe, 0xfa, 0xad, 0xc9, 0xbb, 0x19,
	       0xb3, 0x4a, 0xc8, 0x6b, 0x26, 0x98, 0x4b, 0x63, 0x41, 0x81,
	       0xe1, 0x12, 0xab, 0xcc, 0x89, 0xbe, 0xdf, 0xa8, 0x7e, 0xf5,
	       0x0e, 0x07, 0xf6, 0x92, 0x89, 0x64, 0x3b, 0xc2, 0x64, 0xa2,
	       0x4b, 0xd1, 0x6b, 0x9b, 0x4e, 0x6a, 0xf2, 0x63, 0xf7, 0xc3,
	       0xe0, 0x9f, 0xc5, 0x4e, 0xb6, 0x77, 0x0a, 0xad, 0x6d, 0x0f,
	       0x30, 0x87, 0x6b, 0xfb, 0x66, 0xb3, 0x90, 0x87, 0xa3, 0x48,
	       0xbe, 0xa4, 0x34, 0x9c, 0x5a, 0x93, 0xa3, 0x74, 0x0e, 0x36,
	       0x8e, 0xf6, 0x3b, 0x6c, 0xae, 0xa0, 0x6a, 0xa1, 0x1a, 0x12,
	       0x78, 0x99, 0x75, 0x50, 0xb1, 0x72, 0xed, 0x22, 0x34, 0x0f,
	       0xe1, 0x89, 0xfe, 0x81, 0x0a, 0xcc, 0x2a, 0xd0, 0xf3, 0x25,
	       0xe6, 0xd9, 0x19, 0x06, 0x20, 0x2d, 0x29, 0x8b, 0xdd, 0xb5,
	       0x60, 0xf4, 0x0d, 0x08, 0x97, 0x7b, 0x81, 0x4a, 0xfb, 0x20,
	       0xfb, 0x83, 0xa3, 0xc8, 0x1d, 0x79, 0xb9 ),
	FINGERPRINT ( 0x7e, 0x54, 0x41, 0x60, 0x21, 0xca, 0x3e, 0x63,
		      0xce, 0x5a, 0x41, 0x6c, 0xbe, 0x52, 0x01, 0x88,
		      0xcf, 0x41, 0x36, 0x48, 0xdb, 0xe3, 0xdf, 0x8e,
		      0x79, 0x73, 0x5f, 0xcf, 0x8e, 0x8e, 0xac, 0xd8 ) );

/*
 * subject	www.google.com
 * issuer	Thawte SGC CA
 */
CERTIFICATE ( google_crt,
	DATA ( 0x30, 0x82, 0x03, 0x21, 0x30, 0x82, 0x02, 0x8a, 0xa0, 0x03,
	       0x02, 0x01, 0x02, 0x02, 0x10, 0x4f, 0x9d, 0x96, 0xd9, 0x66,
	       0xb0, 0x99, 0x2b, 0x54, 0xc2, 0x95, 0x7c, 0xb4, 0x15, 0x7d,
	       0x4d, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
	       0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x4c, 0x31, 0x0b,
	       0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x5a,
	       0x41, 0x31, 0x25, 0x30, 0x23, 0x06, 0x03, 0x55, 0x04, 0x0a,
	       0x13, 0x1c, 0x54, 0x68, 0x61, 0x77, 0x74, 0x65, 0x20, 0x43,
	       0x6f, 0x6e, 0x73, 0x75, 0x6c, 0x74, 0x69, 0x6e, 0x67, 0x20,
	       0x28, 0x50, 0x74, 0x79, 0x29, 0x20, 0x4c, 0x74, 0x64, 0x2e,
	       0x31, 0x16, 0x30, 0x14, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13,
	       0x0d, 0x54, 0x68, 0x61, 0x77, 0x74, 0x65, 0x20, 0x53, 0x47,
	       0x43, 0x20, 0x43, 0x41, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x31,
	       0x31, 0x30, 0x32, 0x36, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
	       0x5a, 0x17, 0x0d, 0x31, 0x33, 0x30, 0x39, 0x33, 0x30, 0x32,
	       0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x68, 0x31, 0x0b,
	       0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55,
	       0x53, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08,
	       0x13, 0x0a, 0x43, 0x61, 0x6c, 0x69, 0x66, 0x6f, 0x72, 0x6e,
	       0x69, 0x61, 0x31, 0x16, 0x30, 0x14, 0x06, 0x03, 0x55, 0x04,
	       0x07, 0x14, 0x0d, 0x4d, 0x6f, 0x75, 0x6e, 0x74, 0x61, 0x69,
	       0x6e, 0x20, 0x56, 0x69, 0x65, 0x77, 0x31, 0x13, 0x30, 0x11,
	       0x06, 0x03, 0x55, 0x04, 0x0a, 0x14, 0x0a, 0x47, 0x6f, 0x6f,
	       0x67, 0x6c, 0x65, 0x20, 0x49, 0x6e, 0x63, 0x31, 0x17, 0x30,
	       0x15, 0x06, 0x03, 0x55, 0x04, 0x03, 0x14, 0x0e, 0x77, 0x77,
	       0x77, 0x2e, 0x67, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x2e, 0x63,
	       0x6f, 0x6d, 0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09, 0x2a,
	       0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00,
	       0x03, 0x81, 0x8d, 0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81,
	       0x00, 0xde, 0xb7, 0x26, 0x43, 0xa6, 0x99, 0x85, 0xcd, 0x38,
	       0xa7, 0x15, 0x09, 0xb9, 0xcf, 0x0f, 0xc9, 0xc3, 0x55, 0x8c,
	       0x88, 0xee, 0x8c, 0x8d, 0x28, 0x27, 0x24, 0x4b, 0x2a, 0x5e,
	       0xa0, 0xd8, 0x16, 0xfa, 0x61, 0x18, 0x4b, 0xcf, 0x6d, 0x60,
	       0x80, 0xd3, 0x35, 0x40, 0x32, 0x72, 0xc0, 0x8f, 0x12, 0xd8,
	       0xe5, 0x4e, 0x8f, 0xb9, 0xb2, 0xf6, 0xd9, 0x15, 0x5e, 0x5a,
	       0x86, 0x31, 0xa3, 0xba, 0x86, 0xaa, 0x6b, 0xc8, 0xd9, 0x71,
	       0x8c, 0xcc, 0xcd, 0x27, 0x13, 0x1e, 0x9d, 0x42, 0x5d, 0x38,
	       0xf6, 0xa7, 0xac, 0xef, 0xfa, 0x62, 0xf3, 0x18, 0x81, 0xd4,
	       0x24, 0x46, 0x7f, 0x01, 0x77, 0x7c, 0xc6, 0x2a, 0x89, 0x14,
	       0x99, 0xbb, 0x98, 0x39, 0x1d, 0xa8, 0x19, 0xfb, 0x39, 0x00,
	       0x44, 0x7d, 0x1b, 0x94, 0x6a, 0x78, 0x2d, 0x69, 0xad, 0xc0,
	       0x7a, 0x2c, 0xfa, 0xd0, 0xda, 0x20, 0x12, 0x98, 0xd3, 0x02,
	       0x03, 0x01, 0x00, 0x01, 0xa3, 0x81, 0xe7, 0x30, 0x81, 0xe4,
	       0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff,
	       0x04, 0x02, 0x30, 0x00, 0x30, 0x36, 0x06, 0x03, 0x55, 0x1d,
	       0x1f, 0x04, 0x2f, 0x30, 0x2d, 0x30, 0x2b, 0xa0, 0x29, 0xa0,
	       0x27, 0x86, 0x25, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f,
	       0x63, 0x72, 0x6c, 0x2e, 0x74, 0x68, 0x61, 0x77, 0x74, 0x65,
	       0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x54, 0x68, 0x61, 0x77, 0x74,
	       0x65, 0x53, 0x47, 0x43, 0x43, 0x41, 0x2e, 0x63, 0x72, 0x6c,
	       0x30, 0x28, 0x06, 0x03, 0x55, 0x1d, 0x25, 0x04, 0x21, 0x30,
	       0x1f, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03,
	       0x01, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03,
	       0x02, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x86, 0xf8, 0x42,
	       0x04, 0x01, 0x30, 0x72, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05,
	       0x05, 0x07, 0x01, 0x01, 0x04, 0x66, 0x30, 0x64, 0x30, 0x22,
	       0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x01,
	       0x86, 0x16, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x6f,
	       0x63, 0x73, 0x70, 0x2e, 0x74, 0x68, 0x61, 0x77, 0x74, 0x65,
	       0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x3e, 0x06, 0x08, 0x2b, 0x06,
	       0x01, 0x05, 0x05, 0x07, 0x30, 0x02, 0x86, 0x32, 0x68, 0x74,
	       0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x74,
	       0x68, 0x61, 0x77, 0x74, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x2f,
	       0x72, 0x65, 0x70, 0x6f, 0x73, 0x69, 0x74, 0x6f, 0x72, 0x79,
	       0x2f, 0x54, 0x68, 0x61, 0x77, 0x74, 0x65, 0x5f, 0x53, 0x47,
	       0x43, 0x5f, 0x43, 0x41, 0x2e, 0x63, 0x72, 0x74, 0x30, 0x0d,
	       0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
	       0x05, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00, 0x21, 0xac, 0xd5,
	       0xae, 0xca, 0x34, 0x89, 0x5a, 0xc2, 0xab, 0x52, 0xd2, 0xb2,
	       0x34, 0x66, 0x9d, 0x7a, 0xab, 0xee, 0xe6, 0x7c, 0xd5, 0x7e,
	       0xc2, 0x5c, 0x28, 0xbb, 0x74, 0x00, 0xc9, 0x10, 0x1f, 0x42,
	       0x13, 0xfc, 0x69, 0x8a, 0x1e, 0x24, 0xa0, 0x02, 0x00, 0xe9,
	       0xba, 0x5b, 0xca, 0x19, 0x04, 0xb2, 0xd3, 0xaf, 0x01, 0xb2,
	       0x7e, 0x5f, 0x14, 0xdb, 0xa6, 0xdb, 0x52, 0xb9, 0x9a, 0xf3,
	       0x12, 0x7f, 0x7c, 0xa2, 0x9c, 0x3b, 0x6f, 0x99, 0x7d, 0xea,
	       0x50, 0x0d, 0x76, 0x23, 0x12, 0xff, 0xf7, 0x66, 0x73, 0x29,
	       0xb7, 0x95, 0x0a, 0xad, 0xd8, 0x8b, 0xb2, 0xde, 0x20, 0xe9,
	       0x0a, 0x70, 0x64, 0x11, 0x08, 0xc8, 0x5a, 0xf1, 0x7d, 0x9e,
	       0xec, 0x69, 0xa5, 0xa5, 0xd5, 0x82, 0xd7, 0x27, 0x1e, 0x9e,
	       0x56, 0xcd, 0xd2, 0x76, 0xd5, 0x79, 0x2b, 0xf7, 0x25, 0x43,
	       0x1c, 0x69, 0xf0, 0xb8, 0xf9 ),
	FINGERPRINT ( 0xec, 0x6a, 0x6b, 0x15, 0x6b, 0x30, 0x62, 0xfa,
		      0x99, 0x49, 0x9d, 0x1e, 0x15, 0x15, 0xcf, 0x6c,
		      0x50, 0x48, 0xaf, 0x17, 0x94, 0x57, 0x48, 0x39,
		      0x6b, 0xd2, 0xec, 0xf1, 0x2b, 0x8d, 0xe2, 0x2c ) );

/*
 * subject	VeriSign Class 3 International Server CA - G3
 * issuer	VeriSign Class 3 Public Primary Certification Authority - G5
 */
CERTIFICATE ( verisign_crt,
	DATA ( 0x30, 0x82, 0x06, 0x29, 0x30, 0x82, 0x05, 0x11, 0xa0, 0x03,
	       0x02, 0x01, 0x02, 0x02, 0x10, 0x64, 0x1b, 0xe8, 0x20, 0xce,
	       0x02, 0x08, 0x13, 0xf3, 0x2d, 0x4d, 0x2d, 0x95, 0xd6, 0x7e,
	       0x67, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
	       0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x81, 0xca, 0x31,
	       0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
	       0x55, 0x53, 0x31, 0x17, 0x30, 0x15, 0x06, 0x03, 0x55, 0x04,
	       0x0a, 0x13, 0x0e, 0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67,
	       0x6e, 0x2c, 0x20, 0x49, 0x6e, 0x63, 0x2e, 0x31, 0x1f, 0x30,
	       0x1d, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x13, 0x16, 0x56, 0x65,
	       0x72, 0x69, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x54, 0x72, 0x75,
	       0x73, 0x74, 0x20, 0x4e, 0x65, 0x74, 0x77, 0x6f, 0x72, 0x6b,
	       0x31, 0x3a, 0x30, 0x38, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x13,
	       0x31, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x30, 0x36, 0x20,
	       0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67, 0x6e, 0x2c, 0x20,
	       0x49, 0x6e, 0x63, 0x2e, 0x20, 0x2d, 0x20, 0x46, 0x6f, 0x72,
	       0x20, 0x61, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x7a, 0x65,
	       0x64, 0x20, 0x75, 0x73, 0x65, 0x20, 0x6f, 0x6e, 0x6c, 0x79,
	       0x31, 0x45, 0x30, 0x43, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13,
	       0x3c, 0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67, 0x6e, 0x20,
	       0x43, 0x6c, 0x61, 0x73, 0x73, 0x20, 0x33, 0x20, 0x50, 0x75,
	       0x62, 0x6c, 0x69, 0x63, 0x20, 0x50, 0x72, 0x69, 0x6d, 0x61,
	       0x72, 0x79, 0x20, 0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69,
	       0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x41, 0x75, 0x74,
	       0x68, 0x6f, 0x72, 0x69, 0x74, 0x79, 0x20, 0x2d, 0x20, 0x47,
	       0x35, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x30, 0x30, 0x32, 0x30,
	       0x38, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x17, 0x0d,
	       0x32, 0x30, 0x30, 0x32, 0x30, 0x37, 0x32, 0x33, 0x35, 0x39,
	       0x35, 0x39, 0x5a, 0x30, 0x81, 0xbc, 0x31, 0x0b, 0x30, 0x09,
	       0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31,
	       0x17, 0x30, 0x15, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x0e,
	       0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67, 0x6e, 0x2c, 0x20,
	       0x49, 0x6e, 0x63, 0x2e, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x03,
	       0x55, 0x04, 0x0b, 0x13, 0x16, 0x56, 0x65, 0x72, 0x69, 0x53,
	       0x69, 0x67, 0x6e, 0x20, 0x54, 0x72, 0x75, 0x73, 0x74, 0x20,
	       0x4e, 0x65, 0x74, 0x77, 0x6f, 0x72, 0x6b, 0x31, 0x3b, 0x30,
	       0x39, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x13, 0x32, 0x54, 0x65,
	       0x72, 0x6d, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x75, 0x73, 0x65,
	       0x20, 0x61, 0x74, 0x20, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3a,
	       0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x76, 0x65, 0x72, 0x69,
	       0x73, 0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x72,
	       0x70, 0x61, 0x20, 0x28, 0x63, 0x29, 0x31, 0x30, 0x31, 0x36,
	       0x30, 0x34, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x2d, 0x56,
	       0x65, 0x72, 0x69, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x43, 0x6c,
	       0x61, 0x73, 0x73, 0x20, 0x33, 0x20, 0x49, 0x6e, 0x74, 0x65,
	       0x72, 0x6e, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x61, 0x6c, 0x20,
	       0x53, 0x65, 0x72, 0x76, 0x65, 0x72, 0x20, 0x43, 0x41, 0x20,
	       0x2d, 0x20, 0x47, 0x33, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0d,
	       0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
	       0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30, 0x82,
	       0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0x99, 0xd6, 0x9c,
	       0x62, 0xf0, 0x15, 0xf4, 0x81, 0x9a, 0x41, 0x08, 0x59, 0x8f,
	       0x13, 0x9d, 0x17, 0xc9, 0x9f, 0x51, 0xdc, 0xda, 0xb1, 0x52,
	       0xef, 0xff, 0xe3, 0x41, 0xdd, 0xe0, 0xdf, 0xc4, 0x28, 0xc6,
	       0xe3, 0xad, 0x79, 0x1f, 0x27, 0x10, 0x98, 0xb8, 0xbb, 0x20,
	       0x97, 0xc1, 0x28, 0x44, 0x41, 0x0f, 0xea, 0xa9, 0xa8, 0x52,
	       0xcf, 0x4d, 0x4e, 0x1b, 0x8b, 0xbb, 0xb5, 0xc4, 0x76, 0xd9,
	       0xcc, 0x56, 0x06, 0xee, 0xb3, 0x55, 0x20, 0x2a, 0xde, 0x15,
	       0x8d, 0x71, 0xcb, 0x54, 0xc8, 0x6f, 0x17, 0xcd, 0x89, 0x00,
	       0xe4, 0xdc, 0xff, 0xe1, 0xc0, 0x1f, 0x68, 0x71, 0xe9, 0xc7,
	       0x29, 0x2e, 0x7e, 0xbc, 0x3b, 0xfc, 0xe5, 0xbb, 0xab, 0x26,
	       0x54, 0x8b, 0x66, 0x90, 0xcd, 0xf6, 0x92, 0xb9, 0x31, 0x24,
	       0x80, 0xbc, 0x9e, 0x6c, 0xd5, 0xfc, 0x7e, 0xd2, 0xe1, 0x4b,
	       0x8c, 0xdc, 0x42, 0xfa, 0x44, 0x4b, 0x5f, 0xf8, 0x18, 0xb5,
	       0x2e, 0x30, 0xf4, 0x3d, 0x12, 0x98, 0xd3, 0x62, 0x05, 0x73,
	       0x54, 0xa6, 0x9c, 0xa2, 0x1d, 0xbe, 0x52, 0x83, 0x3a, 0x07,
	       0x46, 0xc4, 0x3b, 0x02, 0x56, 0x21, 0xbf, 0xf2, 0x51, 0x4f,
	       0xd0, 0xa6, 0x99, 0x39, 0xe9, 0xae, 0xa5, 0x3f, 0x89, 0x9b,
	       0x9c, 0x7d, 0xfe, 0x4d, 0x60, 0x07, 0x25, 0x20, 0xf7, 0xbb,
	       0xd7, 0x69, 0x83, 0x2b, 0x82, 0x93, 0x43, 0x37, 0xd9, 0x83,
	       0x41, 0x1b, 0x6b, 0x0b, 0xab, 0x4a, 0x66, 0x84, 0x4f, 0x4a,
	       0x8e, 0xde, 0x7e, 0x34, 0x99, 0x8e, 0x68, 0xd6, 0xca, 0x39,
	       0x06, 0x9b, 0x4c, 0xb3, 0x9a, 0x48, 0x4d, 0x13, 0x46, 0xb4,
	       0x58, 0x21, 0x04, 0xc4, 0xfb, 0xa0, 0x4d, 0xac, 0x2e, 0x4b,
	       0x62, 0x12, 0xe3, 0xfb, 0x4d, 0xf6, 0xc9, 0x51, 0x00, 0x01,
	       0x1f, 0xfc, 0x1e, 0x6a, 0x81, 0x2a, 0x38, 0xe0, 0xb9, 0x4f,
	       0xd6, 0x2d, 0x45, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x82,
	       0x02, 0x15, 0x30, 0x82, 0x02, 0x11, 0x30, 0x12, 0x06, 0x03,
	       0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x08, 0x30, 0x06,
	       0x01, 0x01, 0xff, 0x02, 0x01, 0x00, 0x30, 0x70, 0x06, 0x03,
	       0x55, 0x1d, 0x20, 0x04, 0x69, 0x30, 0x67, 0x30, 0x65, 0x06,
	       0x0b, 0x60, 0x86, 0x48, 0x01, 0x86, 0xf8, 0x45, 0x01, 0x07,
	       0x17, 0x03, 0x30, 0x56, 0x30, 0x28, 0x06, 0x08, 0x2b, 0x06,
	       0x01, 0x05, 0x05, 0x07, 0x02, 0x01, 0x16, 0x1c, 0x68, 0x74,
	       0x74, 0x70, 0x73, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e,
	       0x76, 0x65, 0x72, 0x69, 0x73, 0x69, 0x67, 0x6e, 0x2e, 0x63,
	       0x6f, 0x6d, 0x2f, 0x63, 0x70, 0x73, 0x30, 0x2a, 0x06, 0x08,
	       0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x02, 0x02, 0x30, 0x1e,
	       0x1a, 0x1c, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3a, 0x2f, 0x2f,
	       0x77, 0x77, 0x77, 0x2e, 0x76, 0x65, 0x72, 0x69, 0x73, 0x69,
	       0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x72, 0x70, 0x61,
	       0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff,
	       0x04, 0x04, 0x03, 0x02, 0x01, 0x06, 0x30, 0x6d, 0x06, 0x08,
	       0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x01, 0x0c, 0x04, 0x61,
	       0x30, 0x5f, 0xa1, 0x5d, 0xa0, 0x5b, 0x30, 0x59, 0x30, 0x57,
	       0x30, 0x55, 0x16, 0x09, 0x69, 0x6d, 0x61, 0x67, 0x65, 0x2f,
	       0x67, 0x69, 0x66, 0x30, 0x21, 0x30, 0x1f, 0x30, 0x07, 0x06,
	       0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x04, 0x14, 0x8f, 0xe5,
	       0xd3, 0x1a, 0x86, 0xac, 0x8d, 0x8e, 0x6b, 0xc3, 0xcf, 0x80,
	       0x6a, 0xd4, 0x48, 0x18, 0x2c, 0x7b, 0x19, 0x2e, 0x30, 0x25,
	       0x16, 0x23, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x6c,
	       0x6f, 0x67, 0x6f, 0x2e, 0x76, 0x65, 0x72, 0x69, 0x73, 0x69,
	       0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x76, 0x73, 0x6c,
	       0x6f, 0x67, 0x6f, 0x2e, 0x67, 0x69, 0x66, 0x30, 0x34, 0x06,
	       0x03, 0x55, 0x1d, 0x25, 0x04, 0x2d, 0x30, 0x2b, 0x06, 0x08,
	       0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x06, 0x08,
	       0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x02, 0x06, 0x09,
	       0x60, 0x86, 0x48, 0x01, 0x86, 0xf8, 0x42, 0x04, 0x01, 0x06,
	       0x0a, 0x60, 0x86, 0x48, 0x01, 0x86, 0xf8, 0x45, 0x01, 0x08,
	       0x01, 0x30, 0x34, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05,
	       0x07, 0x01, 0x01, 0x04, 0x28, 0x30, 0x26, 0x30, 0x24, 0x06,
	       0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x01, 0x86,
	       0x18, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x6f, 0x63,
	       0x73, 0x70, 0x2e, 0x76, 0x65, 0x72, 0x69, 0x73, 0x69, 0x67,
	       0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x34, 0x06, 0x03, 0x55,
	       0x1d, 0x1f, 0x04, 0x2d, 0x30, 0x2b, 0x30, 0x29, 0xa0, 0x27,
	       0xa0, 0x25, 0x86, 0x23, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f,
	       0x2f, 0x63, 0x72, 0x6c, 0x2e, 0x76, 0x65, 0x72, 0x69, 0x73,
	       0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x70, 0x63,
	       0x61, 0x33, 0x2d, 0x67, 0x35, 0x2e, 0x63, 0x72, 0x6c, 0x30,
	       0x28, 0x06, 0x03, 0x55, 0x1d, 0x11, 0x04, 0x21, 0x30, 0x1f,
	       0xa4, 0x1d, 0x30, 0x1b, 0x31, 0x19, 0x30, 0x17, 0x06, 0x03,
	       0x55, 0x04, 0x03, 0x13, 0x10, 0x56, 0x65, 0x72, 0x69, 0x53,
	       0x69, 0x67, 0x6e, 0x4d, 0x50, 0x4b, 0x49, 0x2d, 0x32, 0x2d,
	       0x37, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16,
	       0x04, 0x14, 0xd7, 0x9b, 0x7c, 0xd8, 0x22, 0xa0, 0x15, 0xf7,
	       0xdd, 0xad, 0x5f, 0xce, 0x29, 0x9b, 0x58, 0xc3, 0xbc, 0x46,
	       0x00, 0xb5, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04,
	       0x18, 0x30, 0x16, 0x80, 0x14, 0x7f, 0xd3, 0x65, 0xa7, 0xc2,
	       0xdd, 0xec, 0xbb, 0xf0, 0x30, 0x09, 0xf3, 0x43, 0x39, 0xfa,
	       0x02, 0xaf, 0x33, 0x31, 0x33, 0x30, 0x0d, 0x06, 0x09, 0x2a,
	       0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05, 0x00,
	       0x03, 0x82, 0x01, 0x01, 0x00, 0x71, 0xb5, 0x7d, 0x73, 0x52,
	       0x4a, 0xdd, 0xd7, 0x4d, 0x34, 0x2b, 0x2e, 0xaf, 0x94, 0x46,
	       0xa5, 0x49, 0x50, 0x02, 0x4f, 0xf8, 0x2f, 0x17, 0x70, 0xf2,
	       0x13, 0xdc, 0x1f, 0x21, 0x86, 0xaa, 0xc2, 0x4f, 0x7c, 0x37,
	       0x3c, 0xd4, 0x46, 0x78, 0xae, 0x5d, 0x78, 0x6f, 0xd1, 0xba,
	       0x5a, 0xbc, 0x10, 0xab, 0x58, 0x36, 0xc5, 0x8c, 0x62, 0x15,
	       0x45, 0x60, 0x17, 0x21, 0xe2, 0xd5, 0x42, 0xa8, 0x77, 0xa1,
	       0x55, 0xd8, 0x43, 0x04, 0x51, 0xf6, 0x6e, 0xba, 0x48, 0xe6,
	       0x5d, 0x4c, 0xb7, 0x44, 0xd3, 0x3e, 0xa4, 0xd5, 0xd6, 0x33,
	       0x9a, 0x9f, 0x0d, 0xe6, 0xd7, 0x4e, 0x96, 0x44, 0x95, 0x5a,
	       0x6c, 0xd6, 0xa3, 0x16, 0x53, 0x0e, 0x98, 0x43, 0xce, 0xa4,
	       0xb8, 0xc3, 0x66, 0x7a, 0x05, 0x5c, 0x62, 0x10, 0xe8, 0x1b,
	       0x12, 0xdb, 0x7d, 0x2e, 0x76, 0x50, 0xff, 0xdf, 0xd7, 0x6b,
	       0x1b, 0xcc, 0x8a, 0xcc, 0x71, 0xfa, 0xb3, 0x40, 0x56, 0x7c,
	       0x33, 0x7a, 0x77, 0x94, 0x5b, 0xf5, 0x0b, 0x53, 0xfb, 0x0e,
	       0x5f, 0xbc, 0x68, 0xfb, 0xaf, 0x2a, 0xee, 0x30, 0x37, 0x79,
	       0x16, 0x93, 0x25, 0x7f, 0x4d, 0x10, 0xff, 0x57, 0xfb, 0xbf,
	       0x6e, 0x3b, 0x33, 0x21, 0xde, 0x79, 0xdc, 0x86, 0x17, 0x59,
	       0x2d, 0x43, 0x64, 0xb7, 0xa6, 0x66, 0x87, 0xea, 0xbc, 0x96,
	       0x46, 0x19, 0x1a, 0x86, 0x8b, 0x6f, 0xd7, 0xb7, 0x49, 0x00,
	       0x5b, 0xdb, 0xa3, 0xbf, 0x29, 0x9a, 0xee, 0xf7, 0xd3, 0x33,
	       0xae, 0xa3, 0xf4, 0x9e, 0x4c, 0xca, 0x5e, 0x69, 0xd4, 0x1b,
	       0xad, 0xb7, 0x90, 0x77, 0x6a, 0xd8, 0x59, 0x6f, 0x79, 0xab,
	       0x01, 0xfa, 0x55, 0xf0, 0x8a, 0x21, 0x66, 0xe5, 0x65, 0x6e,
	       0xfd, 0x7c, 0xd3, 0xdf, 0x1e, 0xeb, 0x7e, 0x3f, 0x06, 0x90,
	       0xfb, 0x19, 0x0b, 0xd3, 0x06, 0x02, 0x1b, 0x78, 0x43, 0x99,
	       0xa8 ),
	FINGERPRINT ( 0x6e, 0x21, 0x87, 0x6c, 0xf1, 0x63, 0x47, 0x20,
		      0x1f, 0x63, 0x7a, 0x17, 0x8c, 0xb4, 0x2b, 0x17,
		      0x1d, 0x52, 0x37, 0x9a, 0xf7, 0xe5, 0xf5, 0xb6,
		      0xf6, 0x75, 0x5b, 0x3e, 0xe9, 0xbb, 0x2e, 0xd4 ) );

/*
 * subject	Thawte SGC CA
issuer= /C=US/O=VeriSign, Inc./OU=Class 3 Public Primary Certification Authority
 */
CERTIFICATE ( thawte_crt,
	DATA ( 0x30, 0x82, 0x03, 0x23, 0x30, 0x82, 0x02, 0x8c, 0xa0, 0x03,
	       0x02, 0x01, 0x02, 0x02, 0x04, 0x30, 0x00, 0x00, 0x02, 0x30,
	       0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
	       0x01, 0x05, 0x05, 0x00, 0x30, 0x5f, 0x31, 0x0b, 0x30, 0x09,
	       0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31,
	       0x17, 0x30, 0x15, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x0e,
	       0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67, 0x6e, 0x2c, 0x20,
	       0x49, 0x6e, 0x63, 0x2e, 0x31, 0x37, 0x30, 0x35, 0x06, 0x03,
	       0x55, 0x04, 0x0b, 0x13, 0x2e, 0x43, 0x6c, 0x61, 0x73, 0x73,
	       0x20, 0x33, 0x20, 0x50, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x20,
	       0x50, 0x72, 0x69, 0x6d, 0x61, 0x72, 0x79, 0x20, 0x43, 0x65,
	       0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f,
	       0x6e, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x74,
	       0x79, 0x30, 0x1e, 0x17, 0x0d, 0x30, 0x34, 0x30, 0x35, 0x31,
	       0x33, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x17, 0x0d,
	       0x31, 0x34, 0x30, 0x35, 0x31, 0x32, 0x32, 0x33, 0x35, 0x39,
	       0x35, 0x39, 0x5a, 0x30, 0x4c, 0x31, 0x0b, 0x30, 0x09, 0x06,
	       0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x5a, 0x41, 0x31, 0x25,
	       0x30, 0x23, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x1c, 0x54,
	       0x68, 0x61, 0x77, 0x74, 0x65, 0x20, 0x43, 0x6f, 0x6e, 0x73,
	       0x75, 0x6c, 0x74, 0x69, 0x6e, 0x67, 0x20, 0x28, 0x50, 0x74,
	       0x79, 0x29, 0x20, 0x4c, 0x74, 0x64, 0x2e, 0x31, 0x16, 0x30,
	       0x14, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x0d, 0x54, 0x68,
	       0x61, 0x77, 0x74, 0x65, 0x20, 0x53, 0x47, 0x43, 0x20, 0x43,
	       0x41, 0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86,
	       0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03,
	       0x81, 0x8d, 0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00,
	       0xd4, 0xd3, 0x67, 0xd0, 0x8d, 0x15, 0x7f, 0xae, 0xcd, 0x31,
	       0xfe, 0x7d, 0x1d, 0x91, 0xa1, 0x3f, 0x0b, 0x71, 0x3c, 0xac,
	       0xcc, 0xc8, 0x64, 0xfb, 0x63, 0xfc, 0x32, 0x4b, 0x07, 0x94,
	       0xbd, 0x6f, 0x80, 0xba, 0x2f, 0xe1, 0x04, 0x93, 0xc0, 0x33,
	       0xfc, 0x09, 0x33, 0x23, 0xe9, 0x0b, 0x74, 0x2b, 0x71, 0xc4,
	       0x03, 0xc6, 0xd2, 0xcd, 0xe2, 0x2f, 0xf5, 0x09, 0x63, 0xcd,
	       0xff, 0x48, 0xa5, 0x00, 0xbf, 0xe0, 0xe7, 0xf3, 0x88, 0xb7,
	       0x2d, 0x32, 0xde, 0x98, 0x36, 0xe6, 0x0a, 0xad, 0x00, 0x7b,
	       0xc4, 0x64, 0x4a, 0x3b, 0x84, 0x75, 0x03, 0xf2, 0x70, 0x92,
	       0x7d, 0x0e, 0x62, 0xf5, 0x21, 0xab, 0x69, 0x36, 0x84, 0x31,
	       0x75, 0x90, 0xf8, 0xbf, 0xc7, 0x6c, 0x88, 0x1b, 0x06, 0x95,
	       0x7c, 0xc9, 0xe5, 0xa8, 0xde, 0x75, 0xa1, 0x2c, 0x7a, 0x68,
	       0xdf, 0xd5, 0xca, 0x1c, 0x87, 0x58, 0x60, 0x19, 0x02, 0x03,
	       0x01, 0x00, 0x01, 0xa3, 0x81, 0xfe, 0x30, 0x81, 0xfb, 0x30,
	       0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04,
	       0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x00, 0x30,
	       0x0b, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x04, 0x04, 0x03, 0x02,
	       0x01, 0x06, 0x30, 0x11, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01,
	       0x86, 0xf8, 0x42, 0x01, 0x01, 0x04, 0x04, 0x03, 0x02, 0x01,
	       0x06, 0x30, 0x28, 0x06, 0x03, 0x55, 0x1d, 0x11, 0x04, 0x21,
	       0x30, 0x1f, 0xa4, 0x1d, 0x30, 0x1b, 0x31, 0x19, 0x30, 0x17,
	       0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x10, 0x50, 0x72, 0x69,
	       0x76, 0x61, 0x74, 0x65, 0x4c, 0x61, 0x62, 0x65, 0x6c, 0x33,
	       0x2d, 0x31, 0x35, 0x30, 0x31, 0x06, 0x03, 0x55, 0x1d, 0x1f,
	       0x04, 0x2a, 0x30, 0x28, 0x30, 0x26, 0xa0, 0x24, 0xa0, 0x22,
	       0x86, 0x20, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x63,
	       0x72, 0x6c, 0x2e, 0x76, 0x65, 0x72, 0x69, 0x73, 0x69, 0x67,
	       0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x70, 0x63, 0x61, 0x33,
	       0x2e, 0x63, 0x72, 0x6c, 0x30, 0x32, 0x06, 0x08, 0x2b, 0x06,
	       0x01, 0x05, 0x05, 0x07, 0x01, 0x01, 0x04, 0x26, 0x30, 0x24,
	       0x30, 0x22, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07,
	       0x30, 0x01, 0x86, 0x16, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f,
	       0x2f, 0x6f, 0x63, 0x73, 0x70, 0x2e, 0x74, 0x68, 0x61, 0x77,
	       0x74, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x34, 0x06, 0x03,
	       0x55, 0x1d, 0x25, 0x04, 0x2d, 0x30, 0x2b, 0x06, 0x08, 0x2b,
	       0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x06, 0x08, 0x2b,
	       0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x02, 0x06, 0x09, 0x60,
	       0x86, 0x48, 0x01, 0x86, 0xf8, 0x42, 0x04, 0x01, 0x06, 0x0a,
	       0x60, 0x86, 0x48, 0x01, 0x86, 0xf8, 0x45, 0x01, 0x08, 0x01,
	       0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
	       0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00, 0x55,
	       0xac, 0x63, 0xea, 0xde, 0xa1, 0xdd, 0xd2, 0x90, 0x5f, 0x9f,
	       0x0b, 0xce, 0x76, 0xbe, 0x13, 0x51, 0x8f, 0x93, 0xd9, 0x05,
	       0x2b, 0xc8, 0x1b, 0x77, 0x4b, 0xad, 0x69, 0x50, 0xa1, 0xee,
	       0xde, 0xdc, 0xfd, 0xdb, 0x07, 0xe9, 0xe8, 0x39, 0x94, 0xdc,
	       0xab, 0x72, 0x79, 0x2f, 0x06, 0xbf, 0xab, 0x81, 0x70, 0xc4,
	       0xa8, 0xed, 0xea, 0x53, 0x34, 0xed, 0xef, 0x1e, 0x53, 0xd9,
	       0x06, 0xc7, 0x56, 0x2b, 0xd1, 0x5c, 0xf4, 0xd1, 0x8a, 0x8e,
	       0xb4, 0x2b, 0xb1, 0x37, 0x90, 0x48, 0x08, 0x42, 0x25, 0xc5,
	       0x3e, 0x8a, 0xcb, 0x7f, 0xeb, 0x6f, 0x04, 0xd1, 0x6d, 0xc5,
	       0x74, 0xa2, 0xf7, 0xa2, 0x7c, 0x7b, 0x60, 0x3c, 0x77, 0xcd,
	       0x0e, 0xce, 0x48, 0x02, 0x7f, 0x01, 0x2f, 0xb6, 0x9b, 0x37,
	       0xe0, 0x2a, 0x2a, 0x36, 0xdc, 0xd5, 0x85, 0xd6, 0xac, 0xe5,
	       0x3f, 0x54, 0x6f, 0x96, 0x1e, 0x05, 0xaf ),
	FINGERPRINT ( 0x10, 0x85, 0xa6, 0xf4, 0x54, 0xd0, 0xc9, 0x11,
		      0x98, 0xfd, 0xda, 0xb1, 0x1a, 0x31, 0xc7, 0x16,
		      0xd5, 0xdc, 0xd6, 0x8d, 0xf9, 0x1c, 0x03, 0x9c,
		      0xe1, 0x8d, 0xca, 0x9b, 0xeb, 0x3c, 0xde, 0x3d ) );

/*
 * subject	StartCom Class 2 Primary Intermediate Server CA
 * issuer	StartCom Certification Authority
 */
CERTIFICATE ( startssl_crt,
	DATA ( 0x30, 0x82, 0x07, 0xe3, 0x30, 0x82, 0x05, 0xcb, 0xa0, 0x03,
	       0x02, 0x01, 0x02, 0x02, 0x01, 0x0b, 0x30, 0x0d, 0x06, 0x09,
	       0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05,
	       0x00, 0x30, 0x7d, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55,
	       0x04, 0x06, 0x13, 0x02, 0x49, 0x4c, 0x31, 0x16, 0x30, 0x14,
	       0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x0d, 0x53, 0x74, 0x61,
	       0x72, 0x74, 0x43, 0x6f, 0x6d, 0x20, 0x4c, 0x74, 0x64, 0x2e,
	       0x31, 0x2b, 0x30, 0x29, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x13,
	       0x22, 0x53, 0x65, 0x63, 0x75, 0x72, 0x65, 0x20, 0x44, 0x69,
	       0x67, 0x69, 0x74, 0x61, 0x6c, 0x20, 0x43, 0x65, 0x72, 0x74,
	       0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x53, 0x69,
	       0x67, 0x6e, 0x69, 0x6e, 0x67, 0x31, 0x29, 0x30, 0x27, 0x06,
	       0x03, 0x55, 0x04, 0x03, 0x13, 0x20, 0x53, 0x74, 0x61, 0x72,
	       0x74, 0x43, 0x6f, 0x6d, 0x20, 0x43, 0x65, 0x72, 0x74, 0x69,
	       0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x41,
	       0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x74, 0x79, 0x30, 0x1e,
	       0x17, 0x0d, 0x30, 0x37, 0x31, 0x30, 0x32, 0x34, 0x32, 0x30,
	       0x35, 0x37, 0x30, 0x38, 0x5a, 0x17, 0x0d, 0x31, 0x32, 0x31,
	       0x30, 0x32, 0x32, 0x32, 0x30, 0x35, 0x37, 0x30, 0x38, 0x5a,
	       0x30, 0x81, 0x8c, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55,
	       0x04, 0x06, 0x13, 0x02, 0x49, 0x4c, 0x31, 0x16, 0x30, 0x14,
	       0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x0d, 0x53, 0x74, 0x61,
	       0x72, 0x74, 0x43, 0x6f, 0x6d, 0x20, 0x4c, 0x74, 0x64, 0x2e,
	       0x31, 0x2b, 0x30, 0x29, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x13,
	       0x22, 0x53, 0x65, 0x63, 0x75, 0x72, 0x65, 0x20, 0x44, 0x69,
	       0x67, 0x69, 0x74, 0x61, 0x6c, 0x20, 0x43, 0x65, 0x72, 0x74,
	       0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x53, 0x69,
	       0x67, 0x6e, 0x69, 0x6e, 0x67, 0x31, 0x38, 0x30, 0x36, 0x06,
	       0x03, 0x55, 0x04, 0x03, 0x13, 0x2f, 0x53, 0x74, 0x61, 0x72,
	       0x74, 0x43, 0x6f, 0x6d, 0x20, 0x43, 0x6c, 0x61, 0x73, 0x73,
	       0x20, 0x32, 0x20, 0x50, 0x72, 0x69, 0x6d, 0x61, 0x72, 0x79,
	       0x20, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6d, 0x65, 0x64, 0x69,
	       0x61, 0x74, 0x65, 0x20, 0x53, 0x65, 0x72, 0x76, 0x65, 0x72,
	       0x20, 0x43, 0x41, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0d, 0x06,
	       0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
	       0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30, 0x82, 0x01,
	       0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xe2, 0x4f, 0x39, 0x2f,
	       0xa1, 0x8c, 0x9a, 0x85, 0xad, 0x08, 0x0e, 0x08, 0x3e, 0x57,
	       0xf2, 0x88, 0x01, 0x21, 0x1b, 0x94, 0xa9, 0x6c, 0xe2, 0xb8,
	       0xdb, 0xaa, 0x19, 0x18, 0x46, 0x3a, 0x52, 0xa1, 0xf5, 0x0f,
	       0xf4, 0x6e, 0x8c, 0xea, 0x96, 0x8c, 0x96, 0x87, 0x79, 0x13,
	       0x40, 0x51, 0x2f, 0x22, 0xf2, 0x0c, 0x8b, 0x87, 0x0f, 0x65,
	       0xdf, 0x71, 0x74, 0x34, 0x43, 0x55, 0xb1, 0x35, 0x09, 0x9b,
	       0xd9, 0xbc, 0x1f, 0xfa, 0xeb, 0x42, 0xd0, 0x97, 0x40, 0x72,
	       0xb7, 0x43, 0x96, 0x3d, 0xba, 0x96, 0x9d, 0x5d, 0x50, 0x02,
	       0x1c, 0x9b, 0x91, 0x8d, 0x9c, 0xc0, 0xac, 0xd7, 0xbb, 0x2f,
	       0x17, 0xd7, 0xcb, 0x3e, 0x82, 0x9d, 0x73, 0xeb, 0x07, 0x42,
	       0x92, 0xb2, 0xcd, 0x64, 0xb3, 0x74, 0x55, 0x1b, 0xb4, 0x4b,
	       0x86, 0x21, 0x2c, 0xf7, 0x78, 0x87, 0x32, 0xe0, 0x16, 0xe4,
	       0xda, 0xbd, 0x4c, 0x95, 0xea, 0xa4, 0x0a, 0x7e, 0xb6, 0x0a,
	       0x0d, 0x2e, 0x8a, 0xcf, 0x55, 0xab, 0xc3, 0xe5, 0xdd, 0x41,
	       0x8a, 0x4e, 0xe6, 0x6f, 0x65, 0x6c, 0xb2, 0x40, 0xcf, 0x17,
	       0x5d, 0xb9, 0xc3, 0x6a, 0x0b, 0x27, 0x11, 0x84, 0x77, 0x61,
	       0xf6, 0xc2, 0x7c, 0xed, 0xc0, 0x8d, 0x78, 0x14, 0x18, 0x99,
	       0x81, 0x99, 0x75, 0x63, 0xb7, 0xe8, 0x53, 0xd3, 0xba, 0x61,
	       0xe9, 0x0e, 0xfa, 0xa2, 0x30, 0xf3, 0x46, 0xa2, 0xb9, 0xc9,
	       0x1f, 0x6c, 0x80, 0x5a, 0x40, 0xac, 0x27, 0xed, 0x48, 0x47,
	       0x33, 0xb0, 0x54, 0xc6, 0x46, 0x1a, 0xf3, 0x35, 0x61, 0xc1,
	       0x02, 0x29, 0x90, 0x54, 0x7e, 0x64, 0x4d, 0xc4, 0x30, 0x52,
	       0x02, 0x82, 0xd7, 0xdf, 0xce, 0x21, 0x6e, 0x18, 0x91, 0xd7,
	       0xb8, 0xab, 0x8c, 0x27, 0x17, 0xb5, 0xf0, 0xa3, 0x01, 0x2f,
	       0x8e, 0xd2, 0x2e, 0x87, 0x3a, 0x3d, 0xb4, 0x29, 0x67, 0x8a,
	       0xc4, 0x03, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x82, 0x03,
	       0x5c, 0x30, 0x82, 0x03, 0x58, 0x30, 0x0c, 0x06, 0x03, 0x55,
	       0x1d, 0x13, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xff, 0x30,
	       0x0b, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x04, 0x04, 0x03, 0x02,
	       0x01, 0xa6, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04,
	       0x16, 0x04, 0x14, 0x11, 0xdb, 0x23, 0x45, 0xfd, 0x54, 0xcc,
	       0x6a, 0x71, 0x6f, 0x84, 0x8a, 0x03, 0xd7, 0xbe, 0xf7, 0x01,
	       0x2f, 0x26, 0x86, 0x30, 0x81, 0xa8, 0x06, 0x03, 0x55, 0x1d,
	       0x23, 0x04, 0x81, 0xa0, 0x30, 0x81, 0x9d, 0x80, 0x14, 0x4e,
	       0x0b, 0xef, 0x1a, 0xa4, 0x40, 0x5b, 0xa5, 0x17, 0x69, 0x87,
	       0x30, 0xca, 0x34, 0x68, 0x43, 0xd0, 0x41, 0xae, 0xf2, 0xa1,
	       0x81, 0x81, 0xa4, 0x7f, 0x30, 0x7d, 0x31, 0x0b, 0x30, 0x09,
	       0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x49, 0x4c, 0x31,
	       0x16, 0x30, 0x14, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x0d,
	       0x53, 0x74, 0x61, 0x72, 0x74, 0x43, 0x6f, 0x6d, 0x20, 0x4c,
	       0x74, 0x64, 0x2e, 0x31, 0x2b, 0x30, 0x29, 0x06, 0x03, 0x55,
	       0x04, 0x0b, 0x13, 0x22, 0x53, 0x65, 0x63, 0x75, 0x72, 0x65,
	       0x20, 0x44, 0x69, 0x67, 0x69, 0x74, 0x61, 0x6c, 0x20, 0x43,
	       0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65,
	       0x20, 0x53, 0x69, 0x67, 0x6e, 0x69, 0x6e, 0x67, 0x31, 0x29,
	       0x30, 0x27, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x20, 0x53,
	       0x74, 0x61, 0x72, 0x74, 0x43, 0x6f, 0x6d, 0x20, 0x43, 0x65,
	       0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f,
	       0x6e, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x74,
	       0x79, 0x82, 0x01, 0x01, 0x30, 0x09, 0x06, 0x03, 0x55, 0x1d,
	       0x12, 0x04, 0x02, 0x30, 0x00, 0x30, 0x3d, 0x06, 0x08, 0x2b,
	       0x06, 0x01, 0x05, 0x05, 0x07, 0x01, 0x01, 0x04, 0x31, 0x30,
	       0x2f, 0x30, 0x2d, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05,
	       0x07, 0x30, 0x02, 0x86, 0x21, 0x68, 0x74, 0x74, 0x70, 0x3a,
	       0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x73, 0x74, 0x61, 0x72,
	       0x74, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x73,
	       0x66, 0x73, 0x63, 0x61, 0x2e, 0x63, 0x72, 0x74, 0x30, 0x60,
	       0x06, 0x03, 0x55, 0x1d, 0x1f, 0x04, 0x59, 0x30, 0x57, 0x30,
	       0x2c, 0xa0, 0x2a, 0xa0, 0x28, 0x86, 0x26, 0x68, 0x74, 0x74,
	       0x70, 0x3a, 0x2f, 0x2f, 0x63, 0x65, 0x72, 0x74, 0x2e, 0x73,
	       0x74, 0x61, 0x72, 0x74, 0x63, 0x6f, 0x6d, 0x2e, 0x6f, 0x72,
	       0x67, 0x2f, 0x73, 0x66, 0x73, 0x63, 0x61, 0x2d, 0x63, 0x72,
	       0x6c, 0x2e, 0x63, 0x72, 0x6c, 0x30, 0x27, 0xa0, 0x25, 0xa0,
	       0x23, 0x86, 0x21, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f,
	       0x63, 0x72, 0x6c, 0x2e, 0x73, 0x74, 0x61, 0x72, 0x74, 0x73,
	       0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x73, 0x66, 0x73,
	       0x63, 0x61, 0x2e, 0x63, 0x72, 0x6c, 0x30, 0x82, 0x01, 0x5d,
	       0x06, 0x03, 0x55, 0x1d, 0x20, 0x04, 0x82, 0x01, 0x54, 0x30,
	       0x82, 0x01, 0x50, 0x30, 0x82, 0x01, 0x4c, 0x06, 0x0b, 0x2b,
	       0x06, 0x01, 0x04, 0x01, 0x81, 0xb5, 0x37, 0x01, 0x01, 0x04,
	       0x30, 0x82, 0x01, 0x3b, 0x30, 0x2f, 0x06, 0x08, 0x2b, 0x06,
	       0x01, 0x05, 0x05, 0x07, 0x02, 0x01, 0x16, 0x23, 0x68, 0x74,
	       0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x63, 0x65, 0x72, 0x74, 0x2e,
	       0x73, 0x74, 0x61, 0x72, 0x74, 0x63, 0x6f, 0x6d, 0x2e, 0x6f,
	       0x72, 0x67, 0x2f, 0x70, 0x6f, 0x6c, 0x69, 0x63, 0x79, 0x2e,
	       0x70, 0x64, 0x66, 0x30, 0x35, 0x06, 0x08, 0x2b, 0x06, 0x01,
	       0x05, 0x05, 0x07, 0x02, 0x01, 0x16, 0x29, 0x68, 0x74, 0x74,
	       0x70, 0x3a, 0x2f, 0x2f, 0x63, 0x65, 0x72, 0x74, 0x2e, 0x73,
	       0x74, 0x61, 0x72, 0x74, 0x63, 0x6f, 0x6d, 0x2e, 0x6f, 0x72,
	       0x67, 0x2f, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x6d, 0x65, 0x64,
	       0x69, 0x61, 0x74, 0x65, 0x2e, 0x70, 0x64, 0x66, 0x30, 0x81,
	       0xd0, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x02,
	       0x02, 0x30, 0x81, 0xc3, 0x30, 0x27, 0x16, 0x20, 0x53, 0x74,
	       0x61, 0x72, 0x74, 0x20, 0x43, 0x6f, 0x6d, 0x6d, 0x65, 0x72,
	       0x63, 0x69, 0x61, 0x6c, 0x20, 0x28, 0x53, 0x74, 0x61, 0x72,
	       0x74, 0x43, 0x6f, 0x6d, 0x29, 0x20, 0x4c, 0x74, 0x64, 0x2e,
	       0x30, 0x03, 0x02, 0x01, 0x01, 0x1a, 0x81, 0x97, 0x4c, 0x69,
	       0x6d, 0x69, 0x74, 0x65, 0x64, 0x20, 0x4c, 0x69, 0x61, 0x62,
	       0x69, 0x6c, 0x69, 0x74, 0x79, 0x2c, 0x20, 0x72, 0x65, 0x61,
	       0x64, 0x20, 0x74, 0x68, 0x65, 0x20, 0x73, 0x65, 0x63, 0x74,
	       0x69, 0x6f, 0x6e, 0x20, 0x2a, 0x4c, 0x65, 0x67, 0x61, 0x6c,
	       0x20, 0x4c, 0x69, 0x6d, 0x69, 0x74, 0x61, 0x74, 0x69, 0x6f,
	       0x6e, 0x73, 0x2a, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65,
	       0x20, 0x53, 0x74, 0x61, 0x72, 0x74, 0x43, 0x6f, 0x6d, 0x20,
	       0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74,
	       0x69, 0x6f, 0x6e, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72,
	       0x69, 0x74, 0x79, 0x20, 0x50, 0x6f, 0x6c, 0x69, 0x63, 0x79,
	       0x20, 0x61, 0x76, 0x61, 0x69, 0x6c, 0x61, 0x62, 0x6c, 0x65,
	       0x20, 0x61, 0x74, 0x20, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f,
	       0x2f, 0x63, 0x65, 0x72, 0x74, 0x2e, 0x73, 0x74, 0x61, 0x72,
	       0x74, 0x63, 0x6f, 0x6d, 0x2e, 0x6f, 0x72, 0x67, 0x2f, 0x70,
	       0x6f, 0x6c, 0x69, 0x63, 0x79, 0x2e, 0x70, 0x64, 0x66, 0x30,
	       0x11, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x86, 0xf8, 0x42,
	       0x01, 0x01, 0x04, 0x04, 0x03, 0x02, 0x00, 0x07, 0x30, 0x51,
	       0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x86, 0xf8, 0x42, 0x01,
	       0x0d, 0x04, 0x44, 0x16, 0x42, 0x53, 0x74, 0x61, 0x72, 0x74,
	       0x43, 0x6f, 0x6d, 0x20, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x20,
	       0x32, 0x20, 0x50, 0x72, 0x69, 0x6d, 0x61, 0x72, 0x79, 0x20,
	       0x49, 0x6e, 0x74, 0x65, 0x72, 0x6d, 0x65, 0x64, 0x69, 0x61,
	       0x74, 0x65, 0x20, 0x46, 0x72, 0x65, 0x65, 0x20, 0x53, 0x53,
	       0x4c, 0x20, 0x53, 0x65, 0x72, 0x76, 0x65, 0x72, 0x20, 0x43,
	       0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65,
	       0x73, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
	       0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x82, 0x02, 0x01,
	       0x00, 0x64, 0x82, 0x4c, 0x59, 0x5c, 0x0c, 0x6c, 0x16, 0xb6,
	       0xa2, 0x39, 0x45, 0x11, 0x87, 0x98, 0xdf, 0x10, 0x33, 0xae,
	       0x42, 0x9e, 0x25, 0xd1, 0xfd, 0x70, 0x53, 0x45, 0x5e, 0xf4,
	       0x35, 0xc0, 0xe4, 0x30, 0x77, 0xe0, 0x9c, 0xf7, 0xef, 0x27,
	       0xe5, 0x6e, 0x28, 0xef, 0xac, 0xd7, 0xf3, 0x16, 0xcc, 0xc1,
	       0x6c, 0x53, 0x9c, 0x8f, 0x2d, 0x82, 0x6f, 0x94, 0xf7, 0x14,
	       0x36, 0x17, 0xae, 0xb2, 0xe5, 0xa2, 0x3f, 0xa2, 0x83, 0x73,
	       0xa4, 0xdf, 0xa8, 0xbb, 0xca, 0x01, 0x31, 0x6c, 0x9a, 0x43,
	       0x08, 0xdd, 0x1a, 0x5d, 0xad, 0x4b, 0x6d, 0x7f, 0xf4, 0x7e,
	       0xf0, 0x12, 0x3a, 0x05, 0xbf, 0xa7, 0x44, 0x10, 0x07, 0x17,
	       0x44, 0x93, 0x50, 0xe3, 0x95, 0x29, 0xb3, 0xf4, 0x13, 0xd2,
	       0x40, 0xe0, 0x3e, 0xdf, 0x3d, 0x3d, 0x39, 0x5e, 0x81, 0x48,
	       0x0f, 0x32, 0xba, 0x48, 0xd5, 0x03, 0x1b, 0xf5, 0xf7, 0x4e,
	       0x0a, 0x8e, 0x03, 0x8d, 0xfc, 0xca, 0x87, 0xe6, 0xc9, 0xdf,
	       0x26, 0xdf, 0x84, 0x3a, 0x49, 0xb1, 0x99, 0x55, 0x3f, 0xd4,
	       0x2c, 0xab, 0x78, 0x0d, 0x62, 0x03, 0x15, 0x9f, 0xb1, 0x45,
	       0x4b, 0x23, 0x78, 0x62, 0x2a, 0xee, 0xeb, 0x7b, 0x60, 0x2d,
	       0x77, 0x72, 0x1e, 0x61, 0x24, 0x69, 0x62, 0xe8, 0xe1, 0x35,
	       0x3c, 0x82, 0x02, 0x40, 0x15, 0x32, 0x4b, 0x57, 0xcd, 0x97,
	       0xb9, 0x29, 0x8f, 0xa4, 0xd7, 0x84, 0xdb, 0x09, 0xe5, 0x35,
	       0x5b, 0x2f, 0x60, 0x59, 0x49, 0xe9, 0xd6, 0x59, 0xf5, 0xfa,
	       0x5d, 0xc8, 0xcd, 0x4d, 0x94, 0x70, 0xc4, 0xe8, 0x3c, 0x35,
	       0x01, 0x06, 0x75, 0xe8, 0x71, 0x9b, 0x7f, 0x2b, 0x10, 0xca,
	       0x6b, 0x3f, 0xc0, 0xb3, 0x97, 0x72, 0xda, 0xff, 0x74, 0x16,
	       0x8e, 0xfb, 0xfb, 0x9b, 0xe4, 0x05, 0xf0, 0x98, 0xe7, 0x15,
	       0xac, 0xca, 0x24, 0xa2, 0xe7, 0x8a, 0xb4, 0xcf, 0x72, 0x31,
	       0x2b, 0xe3, 0x31, 0x0e, 0x30, 0x5f, 0xd5, 0xcb, 0x94, 0xb3,
	       0xee, 0x35, 0x04, 0xd3, 0x45, 0x07, 0x09, 0x14, 0x51, 0x8a,
	       0xf5, 0x07, 0x31, 0x5e, 0x13, 0x2f, 0x5d, 0x0c, 0xd9, 0xd7,
	       0x36, 0xa8, 0xb0, 0x52, 0x37, 0x94, 0xc2, 0x75, 0x9c, 0x5a,
	       0x2e, 0xa8, 0xd1, 0xb0, 0x73, 0x6b, 0x69, 0xda, 0x0c, 0x3a,
	       0xd6, 0x11, 0x6b, 0xac, 0x9d, 0xbc, 0x45, 0x32, 0x05, 0xe8,
	       0x89, 0x77, 0xd5, 0xb7, 0xbe, 0xc0, 0xed, 0xcd, 0xcd, 0x4e,
	       0x6d, 0x28, 0x58, 0x0a, 0xf0, 0xb4, 0x51, 0xc8, 0xcd, 0x86,
	       0xd1, 0x89, 0x23, 0xc5, 0x23, 0xed, 0xcf, 0x10, 0xf9, 0x0e,
	       0x6e, 0xd8, 0xc1, 0xd7, 0x3f, 0x80, 0xfa, 0xa6, 0x2a, 0x47,
	       0xc4, 0xdb, 0x07, 0x16, 0xe7, 0x0b, 0x8b, 0x53, 0x97, 0x11,
	       0x2e, 0xa3, 0x61, 0x27, 0xb3, 0xbb, 0x4a, 0xd7, 0x01, 0x6d,
	       0xb1, 0x71, 0xa5, 0xe5, 0x23, 0xd6, 0xda, 0xa1, 0xa8, 0x55,
	       0x26, 0xaf, 0xb7, 0xba, 0x5e, 0x56, 0x2d, 0x3d, 0x10, 0xa3,
	       0x71, 0x4a, 0x93, 0x45, 0x1c, 0x7a, 0x3c, 0x92, 0x2a, 0x69,
	       0x3f, 0x4f, 0x1e, 0x96, 0x99, 0x90, 0x09, 0xc8, 0x01, 0xbf,
	       0x28, 0x34, 0xb2, 0x7d, 0x11, 0x50, 0x59, 0xd0, 0x9b, 0xe3,
	       0x33, 0x56, 0x65, 0xee, 0x98, 0x97, 0x3f, 0x3e, 0x54, 0x69,
	       0x2d, 0x26, 0x2e, 0x03, 0x2b, 0xa0, 0xbe, 0xab, 0x6b, 0x80,
	       0x20, 0xe1, 0x3a, 0x4b, 0xbb, 0x26, 0xbe, 0xab, 0xab, 0xa5,
	       0x39, 0xb1, 0x9a, 0x49, 0xa6, 0xb4, 0x5b, 0x96, 0x66, 0xe2,
	       0x49, 0x0b, 0x70, 0x02, 0xaa, 0x49, 0x49, 0x61, 0x0e, 0xfe,
	       0xae, 0xf5, 0xa5, 0x96, 0xe7, 0xf6, 0xba, 0x6c, 0x89, 0x9f,
	       0x0e, 0x72, 0xec, 0xfb, 0x45, 0x00, 0x87, 0x8f, 0x2c, 0xe7,
	       0x7f, 0xb2, 0xc5, 0xa5, 0x52, 0xbe, 0x03, 0xfa, 0x31, 0x2a,
	       0x02, 0x6a, 0xf8 ),
	FINGERPRINT ( 0x9f, 0xd1, 0x93, 0xd3, 0x8f, 0x95, 0x30, 0xab,
		      0x55, 0xa5, 0xc1, 0xb3, 0x51, 0x4a, 0x1d, 0x64,
		      0x1c, 0xec, 0xc2, 0x1c, 0xc5, 0xc2, 0xf5, 0x67,
		      0x48, 0xa7, 0x11, 0x01, 0x69, 0x83, 0xfd, 0x8e ) );

/*
 * subject	RapidSSL SHA256 CA - G3
 * issuer	GeoTrust Global CA
 */
CERTIFICATE ( rapidssl_crt,
	DATA ( 0x30, 0x82, 0x04, 0x25, 0x30, 0x82, 0x03, 0x0d, 0xa0, 0x03,
	       0x02, 0x01, 0x02, 0x02, 0x03, 0x02, 0x3a, 0x77, 0x30, 0x0d,
	       0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
	       0x0b, 0x05, 0x00, 0x30, 0x42, 0x31, 0x0b, 0x30, 0x09, 0x06,
	       0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x16,
	       0x30, 0x14, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x0d, 0x47,
	       0x65, 0x6f, 0x54, 0x72, 0x75, 0x73, 0x74, 0x20, 0x49, 0x6e,
	       0x63, 0x2e, 0x31, 0x1b, 0x30, 0x19, 0x06, 0x03, 0x55, 0x04,
	       0x03, 0x13, 0x12, 0x47, 0x65, 0x6f, 0x54, 0x72, 0x75, 0x73,
	       0x74, 0x20, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x20, 0x43,
	       0x41, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x34, 0x30, 0x38, 0x32,
	       0x39, 0x32, 0x31, 0x33, 0x39, 0x33, 0x32, 0x5a, 0x17, 0x0d,
	       0x32, 0x32, 0x30, 0x35, 0x32, 0x30, 0x32, 0x31, 0x33, 0x39,
	       0x33, 0x32, 0x5a, 0x30, 0x47, 0x31, 0x0b, 0x30, 0x09, 0x06,
	       0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x16,
	       0x30, 0x14, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x0d, 0x47,
	       0x65, 0x6f, 0x54, 0x72, 0x75, 0x73, 0x74, 0x20, 0x49, 0x6e,
	       0x63, 0x2e, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x03, 0x55, 0x04,
	       0x03, 0x13, 0x17, 0x52, 0x61, 0x70, 0x69, 0x64, 0x53, 0x53,
	       0x4c, 0x20, 0x53, 0x48, 0x41, 0x32, 0x35, 0x36, 0x20, 0x43,
	       0x41, 0x20, 0x2d, 0x20, 0x47, 0x33, 0x30, 0x82, 0x01, 0x22,
	       0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
	       0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00,
	       0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xaf,
	       0x54, 0x9b, 0xd9, 0x58, 0x5d, 0x1e, 0x2c, 0x56, 0xc6, 0xd5,
	       0xe8, 0x7f, 0xf4, 0x7d, 0x16, 0x03, 0xff, 0xd0, 0x8b, 0x5a,
	       0xe4, 0x8e, 0xa7, 0xdd, 0x54, 0x2e, 0xd4, 0x04, 0xc0, 0x5d,
	       0x98, 0x9c, 0x8d, 0x90, 0x0f, 0xbc, 0x10, 0x65, 0x5f, 0xda,
	       0x9a, 0xd6, 0x44, 0x7c, 0xc0, 0x9f, 0xb5, 0xe9, 0x4a, 0x8c,
	       0x0b, 0x06, 0x43, 0x04, 0xbb, 0xf4, 0x96, 0xe2, 0x26, 0xf6,
	       0x61, 0x01, 0x91, 0x66, 0x31, 0x22, 0xc3, 0x34, 0x34, 0x5f,
	       0x3f, 0x3f, 0x91, 0x2f, 0x44, 0x5f, 0xdc, 0xc7, 0x14, 0xb6,
	       0x03, 0x9f, 0x86, 0x4b, 0x0e, 0xa3, 0xff, 0xa0, 0x80, 0x02,
	       0x83, 0xc3, 0xd3, 0x1f, 0x69, 0x52, 0xd6, 0x9d, 0x64, 0x0f,
	       0xc9, 0x83, 0xe7, 0x1b, 0xc4, 0x70, 0xac, 0x94, 0xe7, 0xc3,
	       0xa4, 0x6a, 0x2c, 0xbd, 0xb8, 0x9e, 0x69, 0xd8, 0xbe, 0x0a,
	       0x8f, 0x16, 0x63, 0x5a, 0x68, 0x71, 0x80, 0x7b, 0x30, 0xde,
	       0x15, 0x04, 0xbf, 0xcc, 0xd3, 0xbf, 0x3e, 0x48, 0x05, 0x55,
	       0x7a, 0xb3, 0xd7, 0x10, 0x0c, 0x03, 0xfc, 0x9b, 0xfd, 0x08,
	       0xa7, 0x8c, 0x8c, 0xdb, 0xa7, 0x8e, 0xf1, 0x1e, 0x63, 0xdc,
	       0xb3, 0x01, 0x2f, 0x7f, 0xaf, 0x57, 0xc3, 0x3c, 0x48, 0xa7,
	       0x83, 0x68, 0x21, 0xa7, 0x2f, 0xe7, 0xa7, 0x3f, 0xf0, 0xb5,
	       0x0c, 0xfc, 0xf5, 0x84, 0xd1, 0x53, 0xbc, 0x0e, 0x72, 0x4f,
	       0x60, 0x0c, 0x42, 0xb8, 0x98, 0xad, 0x19, 0x88, 0x57, 0xd7,
	       0x04, 0xec, 0x87, 0xbf, 0x7e, 0x87, 0x4e, 0xa3, 0x21, 0xf9,
	       0x53, 0xfd, 0x36, 0x98, 0x48, 0x8d, 0xd6, 0xf8, 0xbb, 0x48,
	       0xf2, 0x29, 0xc8, 0x64, 0xd1, 0xcc, 0x54, 0x48, 0x53, 0x8b,
	       0xaf, 0xb7, 0x65, 0x1e, 0xbf, 0x29, 0x33, 0x29, 0xd9, 0x29,
	       0x60, 0x48, 0xf8, 0xff, 0x91, 0xbc, 0x57, 0x58, 0xe5, 0x35,
	       0x2e, 0xbb, 0x69, 0xb6, 0x59, 0x02, 0x03, 0x01, 0x00, 0x01,
	       0xa3, 0x82, 0x01, 0x1d, 0x30, 0x82, 0x01, 0x19, 0x30, 0x1f,
	       0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80,
	       0x14, 0xc0, 0x7a, 0x98, 0x68, 0x8d, 0x89, 0xfb, 0xab, 0x05,
	       0x64, 0x0c, 0x11, 0x7d, 0xaa, 0x7d, 0x65, 0xb8, 0xca, 0xcc,
	       0x4e, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16,
	       0x04, 0x14, 0xc3, 0x9c, 0xf3, 0xfc, 0xd3, 0x46, 0x08, 0x34,
	       0xbb, 0xce, 0x46, 0x7f, 0xa0, 0x7c, 0x5b, 0xf3, 0xe2, 0x08,
	       0xcb, 0x59, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01,
	       0x01, 0xff, 0x04, 0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02,
	       0x01, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01,
	       0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x01, 0x06, 0x30, 0x35,
	       0x06, 0x03, 0x55, 0x1d, 0x1f, 0x04, 0x2e, 0x30, 0x2c, 0x30,
	       0x2a, 0xa0, 0x28, 0xa0, 0x26, 0x86, 0x24, 0x68, 0x74, 0x74,
	       0x70, 0x3a, 0x2f, 0x2f, 0x67, 0x2e, 0x73, 0x79, 0x6d, 0x63,
	       0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x63, 0x72, 0x6c, 0x73,
	       0x2f, 0x67, 0x74, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x2e,
	       0x63, 0x72, 0x6c, 0x30, 0x2e, 0x06, 0x08, 0x2b, 0x06, 0x01,
	       0x05, 0x05, 0x07, 0x01, 0x01, 0x04, 0x22, 0x30, 0x20, 0x30,
	       0x1e, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30,
	       0x01, 0x86, 0x12, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f,
	       0x67, 0x2e, 0x73, 0x79, 0x6d, 0x63, 0x64, 0x2e, 0x63, 0x6f,
	       0x6d, 0x30, 0x4c, 0x06, 0x03, 0x55, 0x1d, 0x20, 0x04, 0x45,
	       0x30, 0x43, 0x30, 0x41, 0x06, 0x0a, 0x60, 0x86, 0x48, 0x01,
	       0x86, 0xf8, 0x45, 0x01, 0x07, 0x36, 0x30, 0x33, 0x30, 0x31,
	       0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x02, 0x01,
	       0x16, 0x25, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77,
	       0x77, 0x77, 0x2e, 0x67, 0x65, 0x6f, 0x74, 0x72, 0x75, 0x73,
	       0x74, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x72, 0x65, 0x73, 0x6f,
	       0x75, 0x72, 0x63, 0x65, 0x73, 0x2f, 0x63, 0x70, 0x73, 0x30,
	       0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
	       0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0xa3,
	       0x58, 0x1e, 0xc6, 0x43, 0x32, 0xac, 0xac, 0x2f, 0x93, 0x78,
	       0xb7, 0xea, 0xae, 0x54, 0x40, 0x47, 0x2d, 0x7e, 0x78, 0x8d,
	       0x50, 0xf6, 0xf8, 0x66, 0xac, 0xd6, 0x4f, 0x73, 0xd6, 0x44,
	       0xef, 0xaf, 0x0b, 0xcc, 0x5b, 0xc1, 0xf4, 0x4f, 0x9a, 0x8f,
	       0x49, 0x7e, 0x60, 0xaf, 0xc2, 0x27, 0xc7, 0x16, 0xf1, 0xfb,
	       0x93, 0x81, 0x90, 0xa9, 0x7c, 0xef, 0x6f, 0x7e, 0x6e, 0x45,
	       0x94, 0x16, 0x84, 0xbd, 0xec, 0x49, 0xf1, 0xc4, 0x0e, 0xf4,
	       0xaf, 0x04, 0x59, 0x83, 0x87, 0x0f, 0x2c, 0x3b, 0x97, 0xc3,
	       0x5a, 0x12, 0x9b, 0x7b, 0x04, 0x35, 0x7b, 0xa3, 0x95, 0x33,
	       0x08, 0x7b, 0x93, 0x71, 0x22, 0x42, 0xb3, 0xa9, 0xd9, 0x6f,
	       0x4f, 0x81, 0x92, 0xfc, 0x07, 0xb6, 0x79, 0xbc, 0x84, 0x4a,
	       0x9d, 0x77, 0x09, 0xf1, 0xc5, 0x89, 0xf2, 0xf0, 0xb4, 0x9c,
	       0x54, 0xaa, 0x12, 0x7b, 0x0d, 0xba, 0x4f, 0xef, 0x93, 0x19,
	       0xec, 0xef, 0x7d, 0x4e, 0x61, 0xa3, 0x8e, 0x76, 0x9c, 0x59,
	       0xcf, 0x8c, 0x94, 0xb1, 0x84, 0x97, 0xf7, 0x1a, 0xb9, 0x07,
	       0xb8, 0xb2, 0xc6, 0x4f, 0x13, 0x79, 0xdb, 0xbf, 0x4f, 0x51,
	       0x1b, 0x7f, 0x69, 0x0d, 0x51, 0x2a, 0xc1, 0xd6, 0x15, 0xff,
	       0x37, 0x51, 0x34, 0x65, 0x51, 0xf4, 0x1e, 0xbe, 0x38, 0x6a,
	       0xec, 0x0e, 0xab, 0xbf, 0x3d, 0x7b, 0x39, 0x05, 0x7b, 0xf4,
	       0xf3, 0xfb, 0x1a, 0xa1, 0xd0, 0xc8, 0x7e, 0x4e, 0x64, 0x8d,
	       0xcd, 0x8c, 0x61, 0x55, 0x90, 0xfe, 0x3a, 0xca, 0x5d, 0x25,
	       0x0f, 0xf8, 0x1d, 0xa3, 0x4a, 0x74, 0x56, 0x4f, 0x1a, 0x55,
	       0x40, 0x70, 0x75, 0x25, 0xa6, 0x33, 0x2e, 0xba, 0x4b, 0xa5,
	       0x5d, 0x53, 0x9a, 0x0d, 0x30, 0xe1, 0x8d, 0x5f, 0x61, 0x2c,
	       0xaf, 0xcc, 0xef, 0xb0, 0x99, 0xa1, 0x80, 0xff, 0x0b, 0xf2,
	       0x62, 0x4c, 0x70, 0x26, 0x98 ),
	FINGERPRINT ( 0xbc, 0x3f, 0x03, 0xa4, 0x36, 0x24, 0x0e, 0xdb,
		      0xa5, 0xf8, 0x37, 0x14, 0xf6, 0xf6, 0x77, 0xe3,
		      0x4b, 0x37, 0xf9, 0xb1, 0xf0, 0xc0, 0x8c, 0x1e,
		      0x55, 0x8d, 0x98, 0x1e, 0x27, 0x9e, 0x82, 0x09 ) );

/*
 * subject	*.vultr.com
 * issuer	RapidSSL SHA256 CA - G3
 */
CERTIFICATE ( vultr_crt,
	DATA ( 0x30, 0x82, 0x04, 0xa8, 0x30, 0x82, 0x03, 0x90, 0xa0, 0x03,
	       0x02, 0x01, 0x02, 0x02, 0x03, 0x00, 0x95, 0x4d, 0x30, 0x0d,
	       0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
	       0x0b, 0x05, 0x00, 0x30, 0x47, 0x31, 0x0b, 0x30, 0x09, 0x06,
	       0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x16,
	       0x30, 0x14, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x0d, 0x47,
	       0x65, 0x6f, 0x54, 0x72, 0x75, 0x73, 0x74, 0x20, 0x49, 0x6e,
	       0x63, 0x2e, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x03, 0x55, 0x04,
	       0x03, 0x13, 0x17, 0x52, 0x61, 0x70, 0x69, 0x64, 0x53, 0x53,
	       0x4c, 0x20, 0x53, 0x48, 0x41, 0x32, 0x35, 0x36, 0x20, 0x43,
	       0x41, 0x20, 0x2d, 0x20, 0x47, 0x33, 0x30, 0x1e, 0x17, 0x0d,
	       0x31, 0x34, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x33, 0x31,
	       0x32, 0x38, 0x5a, 0x17, 0x0d, 0x31, 0x37, 0x31, 0x32, 0x32,
	       0x32, 0x31, 0x39, 0x32, 0x39, 0x31, 0x30, 0x5a, 0x30, 0x81,
	       0x8f, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x0b,
	       0x13, 0x0a, 0x47, 0x54, 0x36, 0x32, 0x30, 0x37, 0x39, 0x37,
	       0x32, 0x31, 0x31, 0x31, 0x30, 0x2f, 0x06, 0x03, 0x55, 0x04,
	       0x0b, 0x13, 0x28, 0x53, 0x65, 0x65, 0x20, 0x77, 0x77, 0x77,
	       0x2e, 0x72, 0x61, 0x70, 0x69, 0x64, 0x73, 0x73, 0x6c, 0x2e,
	       0x63, 0x6f, 0x6d, 0x2f, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72,
	       0x63, 0x65, 0x73, 0x2f, 0x63, 0x70, 0x73, 0x20, 0x28, 0x63,
	       0x29, 0x31, 0x33, 0x31, 0x2f, 0x30, 0x2d, 0x06, 0x03, 0x55,
	       0x04, 0x0b, 0x13, 0x26, 0x44, 0x6f, 0x6d, 0x61, 0x69, 0x6e,
	       0x20, 0x43, 0x6f, 0x6e, 0x74, 0x72, 0x6f, 0x6c, 0x20, 0x56,
	       0x61, 0x6c, 0x69, 0x64, 0x61, 0x74, 0x65, 0x64, 0x20, 0x2d,
	       0x20, 0x52, 0x61, 0x70, 0x69, 0x64, 0x53, 0x53, 0x4c, 0x28,
	       0x52, 0x29, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04,
	       0x03, 0x0c, 0x0b, 0x2a, 0x2e, 0x76, 0x75, 0x6c, 0x74, 0x72,
	       0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0d,
	       0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
	       0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30, 0x82,
	       0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0x9f, 0xa8, 0x2e,
	       0x65, 0x41, 0x05, 0xec, 0xef, 0x69, 0x92, 0xf6, 0xd3, 0x53,
	       0x4f, 0xd4, 0x8e, 0xd3, 0x49, 0x8c, 0xc7, 0x85, 0x6d, 0xb0,
	       0x71, 0xe0, 0x28, 0x04, 0x80, 0x85, 0x82, 0x3e, 0x3f, 0xdb,
	       0x0c, 0xed, 0xcd, 0x2b, 0x04, 0xc8, 0x67, 0x15, 0x8c, 0x25,
	       0xd6, 0x7a, 0x54, 0x67, 0x94, 0xbe, 0x33, 0x12, 0x33, 0x88,
	       0xfe, 0x5d, 0x1d, 0x36, 0x62, 0x4e, 0xbc, 0x1e, 0x7e, 0xd3,
	       0x3e, 0x7c, 0x3c, 0x59, 0x4a, 0x99, 0x0b, 0xca, 0x9b, 0x1e,
	       0xc7, 0xf7, 0xe7, 0x6d, 0xdc, 0x57, 0xbe, 0x3a, 0xab, 0xc2,
	       0x0b, 0xb1, 0xbe, 0x90, 0xa1, 0x54, 0x07, 0xc5, 0x48, 0x65,
	       0xc1, 0x32, 0x99, 0x92, 0x26, 0x97, 0x90, 0x3e, 0x68, 0x6b,
	       0xac, 0xbd, 0x4f, 0x0e, 0x88, 0x38, 0xd3, 0xdc, 0x80, 0x9e,
	       0xb9, 0x66, 0x5d, 0xeb, 0x19, 0xfd, 0x85, 0xff, 0xba, 0xf0,
	       0x89, 0x20, 0x08, 0xea, 0xd8, 0x01, 0x76, 0x29, 0xdc, 0xf0,
	       0x1c, 0xfa, 0xbf, 0x6f, 0x7b, 0x67, 0xf4, 0xc4, 0xee, 0xe3,
	       0xde, 0x95, 0xa2, 0x76, 0x65, 0x98, 0xc5, 0x21, 0xdc, 0xe9,
	       0x95, 0xee, 0x83, 0x97, 0xae, 0xdd, 0xab, 0xdb, 0xc0, 0x47,
	       0xc8, 0x68, 0x00, 0xb3, 0xab, 0x11, 0x4f, 0x81, 0xf5, 0x45,
	       0x01, 0xd1, 0x64, 0xfd, 0x53, 0xbf, 0x86, 0xef, 0x87, 0xca,
	       0x8e, 0x55, 0xa0, 0x27, 0x27, 0xe5, 0x9e, 0xc1, 0x69, 0x28,
	       0x2a, 0x9f, 0x2d, 0xe2, 0x2c, 0x25, 0xef, 0x74, 0x1b, 0x52,
	       0xe4, 0x81, 0xf4, 0xc2, 0x71, 0x0a, 0x48, 0xff, 0x47, 0xa5,
	       0xea, 0x0a, 0xf5, 0xb1, 0x6d, 0xae, 0x09, 0x2b, 0xf9, 0x23,
	       0x6a, 0x28, 0x58, 0x3d, 0xbb, 0x9f, 0x80, 0xb2, 0x81, 0x03,
	       0xae, 0xe5, 0xea, 0xbe, 0x97, 0xae, 0xec, 0x3c, 0x33, 0xc7,
	       0x68, 0xf0, 0x6c, 0x89, 0x9d, 0x01, 0x2a, 0x1e, 0x2b, 0xd7,
	       0x93, 0x5a, 0xa9, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x82,
	       0x01, 0x52, 0x30, 0x82, 0x01, 0x4e, 0x30, 0x1f, 0x06, 0x03,
	       0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xc3,
	       0x9c, 0xf3, 0xfc, 0xd3, 0x46, 0x08, 0x34, 0xbb, 0xce, 0x46,
	       0x7f, 0xa0, 0x7c, 0x5b, 0xf3, 0xe2, 0x08, 0xcb, 0x59, 0x30,
	       0x57, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x01,
	       0x01, 0x04, 0x4b, 0x30, 0x49, 0x30, 0x1f, 0x06, 0x08, 0x2b,
	       0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x01, 0x86, 0x13, 0x68,
	       0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x67, 0x76, 0x2e, 0x73,
	       0x79, 0x6d, 0x63, 0x64, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x26,
	       0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x02,
	       0x86, 0x1a, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x67,
	       0x76, 0x2e, 0x73, 0x79, 0x6d, 0x63, 0x62, 0x2e, 0x63, 0x6f,
	       0x6d, 0x2f, 0x67, 0x76, 0x2e, 0x63, 0x72, 0x74, 0x30, 0x0e,
	       0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04,
	       0x03, 0x02, 0x05, 0xa0, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d,
	       0x25, 0x04, 0x16, 0x30, 0x14, 0x06, 0x08, 0x2b, 0x06, 0x01,
	       0x05, 0x05, 0x07, 0x03, 0x01, 0x06, 0x08, 0x2b, 0x06, 0x01,
	       0x05, 0x05, 0x07, 0x03, 0x02, 0x30, 0x21, 0x06, 0x03, 0x55,
	       0x1d, 0x11, 0x04, 0x1a, 0x30, 0x18, 0x82, 0x0b, 0x2a, 0x2e,
	       0x76, 0x75, 0x6c, 0x74, 0x72, 0x2e, 0x63, 0x6f, 0x6d, 0x82,
	       0x09, 0x76, 0x75, 0x6c, 0x74, 0x72, 0x2e, 0x63, 0x6f, 0x6d,
	       0x30, 0x2b, 0x06, 0x03, 0x55, 0x1d, 0x1f, 0x04, 0x24, 0x30,
	       0x22, 0x30, 0x20, 0xa0, 0x1e, 0xa0, 0x1c, 0x86, 0x1a, 0x68,
	       0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x67, 0x76, 0x2e, 0x73,
	       0x79, 0x6d, 0x63, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x67,
	       0x76, 0x2e, 0x63, 0x72, 0x6c, 0x30, 0x0c, 0x06, 0x03, 0x55,
	       0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30,
	       0x45, 0x06, 0x03, 0x55, 0x1d, 0x20, 0x04, 0x3e, 0x30, 0x3c,
	       0x30, 0x3a, 0x06, 0x0a, 0x60, 0x86, 0x48, 0x01, 0x86, 0xf8,
	       0x45, 0x01, 0x07, 0x36, 0x30, 0x2c, 0x30, 0x2a, 0x06, 0x08,
	       0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x02, 0x01, 0x16, 0x1e,
	       0x68, 0x74, 0x74, 0x70, 0x73, 0x3a, 0x2f, 0x2f, 0x77, 0x77,
	       0x77, 0x2e, 0x72, 0x61, 0x70, 0x69, 0x64, 0x73, 0x73, 0x6c,
	       0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x6c, 0x65, 0x67, 0x61, 0x6c,
	       0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
	       0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00,
	       0x93, 0x63, 0x02, 0x6a, 0xa1, 0x2a, 0xf3, 0xbe, 0x64, 0x2b,
	       0x36, 0xaf, 0x66, 0x16, 0x49, 0x29, 0x56, 0x6c, 0xc7, 0x75,
	       0x74, 0xf3, 0x69, 0xd8, 0x85, 0xce, 0x50, 0x3b, 0x43, 0x89,
	       0xaf, 0x74, 0x99, 0x26, 0x34, 0xa4, 0x27, 0xa8, 0xfa, 0xfe,
	       0x45, 0xaf, 0xbe, 0x75, 0x22, 0x3d, 0x15, 0xca, 0xa6, 0x38,
	       0xc9, 0x2b, 0x3c, 0xf4, 0x11, 0x9f, 0x96, 0xa7, 0x69, 0x3e,
	       0xf8, 0xf0, 0x88, 0xd8, 0xd2, 0x9c, 0x1c, 0x0e, 0x4c, 0xfd,
	       0xf3, 0x3b, 0x48, 0x25, 0x68, 0xb3, 0x8d, 0x7e, 0x26, 0x73,
	       0x73, 0xcb, 0x3a, 0x41, 0x92, 0x16, 0x55, 0xe1, 0xff, 0x00,
	       0x07, 0xa2, 0xfe, 0xfe, 0x25, 0xfc, 0x86, 0x0f, 0x49, 0xff,
	       0x96, 0x78, 0x02, 0x65, 0xd7, 0xad, 0xcd, 0x0c, 0x82, 0x35,
	       0x56, 0xfe, 0x12, 0x25, 0xa9, 0x8f, 0xc2, 0xa4, 0xe9, 0x43,
	       0xbb, 0x85, 0x62, 0x21, 0x62, 0x5d, 0x70, 0x76, 0x79, 0x75,
	       0xeb, 0xd6, 0x82, 0x53, 0x0d, 0xde, 0x77, 0x95, 0x56, 0x87,
	       0x44, 0x13, 0x82, 0x7d, 0xa9, 0x9a, 0x94, 0x7e, 0x1c, 0x6d,
	       0x7f, 0x72, 0xef, 0x04, 0x84, 0xdf, 0x65, 0x98, 0x17, 0xb4,
	       0xbe, 0xfe, 0x30, 0x0f, 0xfa, 0x8d, 0x9f, 0x29, 0x1d, 0xbd,
	       0x99, 0xa7, 0xe3, 0x09, 0x1d, 0x13, 0x21, 0xfd, 0x9e, 0x03,
	       0x17, 0xb9, 0x9e, 0x48, 0x35, 0x66, 0xe5, 0x86, 0x0a, 0x0f,
	       0x04, 0xfd, 0xcd, 0x3b, 0x13, 0x59, 0xd6, 0x0c, 0x05, 0x8c,
	       0xd2, 0x6b, 0x5c, 0x45, 0x33, 0x43, 0x91, 0xac, 0xb8, 0xdd,
	       0xe3, 0xbe, 0xdf, 0x7b, 0x5c, 0x94, 0xa9, 0xfd, 0xc0, 0xf8,
	       0xa9, 0xd2, 0x82, 0xcd, 0xbf, 0x60, 0xd7, 0xf8, 0x3d, 0x53,
	       0xf7, 0x6a, 0xdc, 0x46, 0xc4, 0x22, 0x84, 0x98, 0x6a, 0x32,
	       0xf2, 0xdf, 0x43, 0xd9, 0x5a, 0xdb, 0x97, 0x20, 0x05, 0x0b,
	       0x3e, 0x06, 0x38, 0x13, 0x3a, 0x21 ),
	FINGERPRINT ( 0x2c, 0x58, 0x63, 0x6e, 0xf1, 0xab, 0x8f, 0xff,
		      0x86, 0x5d, 0x4f, 0x8d, 0x3f, 0xa9, 0x4d, 0x63,
		      0xe7, 0xe6, 0xc6, 0x03, 0x1e, 0xc9, 0x0e, 0xfb,
		      0xe8, 0xaa, 0xa6, 0xf4, 0x6f, 0x21, 0xc7, 0x7b ) );

OCSP ( barclays_ocsp, &barclays_crt, &verisign_crt,
	DATA ( 0x30, 0x51, 0x30, 0x4f, 0x30, 0x4d, 0x30, 0x4b, 0x30, 0x49,
	       0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x05,
	       0x00, 0x04, 0x14, 0xd4, 0xb4, 0x3b, 0x8e, 0x3d, 0x02, 0x49,
	       0x1a, 0x65, 0x50, 0x6f, 0x96, 0x73, 0x14, 0xdd, 0xe8, 0x59,
	       0x44, 0x52, 0xe4, 0x04, 0x14, 0xd7, 0x9b, 0x7c, 0xd8, 0x22,
	       0xa0, 0x15, 0xf7, 0xdd, 0xad, 0x5f, 0xce, 0x29, 0x9b, 0x58,
	       0xc3, 0xbc, 0x46, 0x00, 0xb5, 0x02, 0x10, 0x49, 0x83, 0xfc,
	       0x05, 0x76, 0xdf, 0x36, 0x91, 0x7c, 0x64, 0x2a, 0x27, 0xc1,
	       0xf1, 0x48, 0xe3 ),
	DATA ( 0x30, 0x82, 0x06, 0x4d, 0x0a, 0x01, 0x00, 0xa0, 0x82, 0x06,
	       0x46, 0x30, 0x82, 0x06, 0x42, 0x06, 0x09, 0x2b, 0x06, 0x01,
	       0x05, 0x05, 0x07, 0x30, 0x01, 0x01, 0x04, 0x82, 0x06, 0x33,
	       0x30, 0x82, 0x06, 0x2f, 0x30, 0x82, 0x01, 0x1f, 0xa0, 0x03,
	       0x02, 0x01, 0x00, 0xa1, 0x81, 0x91, 0x30, 0x81, 0x8e, 0x31,
	       0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
	       0x55, 0x53, 0x31, 0x17, 0x30, 0x15, 0x06, 0x03, 0x55, 0x04,
	       0x0a, 0x13, 0x0e, 0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67,
	       0x6e, 0x2c, 0x20, 0x49, 0x6e, 0x63, 0x2e, 0x31, 0x1f, 0x30,
	       0x1d, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x13, 0x16, 0x56, 0x65,
	       0x72, 0x69, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x54, 0x72, 0x75,
	       0x73, 0x74, 0x20, 0x4e, 0x65, 0x74, 0x77, 0x6f, 0x72, 0x6b,
	       0x31, 0x45, 0x30, 0x43, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13,
	       0x3c, 0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67, 0x6e, 0x20,
	       0x43, 0x6c, 0x61, 0x73, 0x73, 0x20, 0x33, 0x20, 0x49, 0x6e,
	       0x74, 0x65, 0x72, 0x6e, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x61,
	       0x6c, 0x20, 0x53, 0x65, 0x72, 0x76, 0x65, 0x72, 0x20, 0x43,
	       0x41, 0x20, 0x2d, 0x20, 0x47, 0x33, 0x20, 0x4f, 0x43, 0x53,
	       0x50, 0x20, 0x52, 0x65, 0x73, 0x70, 0x6f, 0x6e, 0x64, 0x65,
	       0x72, 0x18, 0x0f, 0x32, 0x30, 0x31, 0x32, 0x30, 0x35, 0x31,
	       0x34, 0x31, 0x34, 0x30, 0x31, 0x34, 0x31, 0x5a, 0x30, 0x73,
	       0x30, 0x71, 0x30, 0x49, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e,
	       0x03, 0x02, 0x1a, 0x05, 0x00, 0x04, 0x14, 0xd4, 0xb4, 0x3b,
	       0x8e, 0x3d, 0x02, 0x49, 0x1a, 0x65, 0x50, 0x6f, 0x96, 0x73,
	       0x14, 0xdd, 0xe8, 0x59, 0x44, 0x52, 0xe4, 0x04, 0x14, 0xd7,
	       0x9b, 0x7c, 0xd8, 0x22, 0xa0, 0x15, 0xf7, 0xdd, 0xad, 0x5f,
	       0xce, 0x29, 0x9b, 0x58, 0xc3, 0xbc, 0x46, 0x00, 0xb5, 0x02,
	       0x10, 0x49, 0x83, 0xfc, 0x05, 0x76, 0xdf, 0x36, 0x91, 0x7c,
	       0x64, 0x2a, 0x27, 0xc1, 0xf1, 0x48, 0xe3, 0x80, 0x00, 0x18,
	       0x0f, 0x32, 0x30, 0x31, 0x32, 0x30, 0x35, 0x31, 0x34, 0x31,
	       0x34, 0x30, 0x31, 0x34, 0x31, 0x5a, 0xa0, 0x11, 0x18, 0x0f,
	       0x32, 0x30, 0x31, 0x32, 0x30, 0x35, 0x32, 0x31, 0x31, 0x34,
	       0x30, 0x31, 0x34, 0x31, 0x5a, 0x30, 0x0d, 0x06, 0x09, 0x2a,
	       0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05, 0x00,
	       0x03, 0x81, 0x81, 0x00, 0x78, 0xa8, 0xe7, 0xdc, 0xaf, 0xf2,
	       0x8b, 0xc2, 0x9f, 0x99, 0xf9, 0x44, 0x84, 0xa8, 0x8a, 0x2c,
	       0x35, 0xd7, 0x91, 0xd8, 0x44, 0x3e, 0xd6, 0xe0, 0xc6, 0x3f,
	       0xbf, 0xe3, 0x71, 0x22, 0x45, 0x5a, 0x73, 0xf4, 0x59, 0x1b,
	       0x4c, 0xa5, 0x51, 0x21, 0x5c, 0xfa, 0xbe, 0x10, 0xcb, 0x5f,
	       0xb0, 0x2f, 0xef, 0x22, 0x89, 0xf6, 0xc7, 0x0f, 0x7c, 0x6d,
	       0x26, 0xbe, 0x83, 0x99, 0xe2, 0x02, 0xf5, 0x95, 0x6e, 0xca,
	       0xd9, 0x6d, 0xdd, 0xc2, 0xf0, 0xf2, 0x4f, 0x99, 0x66, 0x93,
	       0x6e, 0x2e, 0xcf, 0xc5, 0xab, 0x68, 0x5f, 0xde, 0x52, 0x3d,
	       0xf2, 0x23, 0x9d, 0xe8, 0x99, 0xd5, 0xf4, 0x4f, 0x8a, 0xf3,
	       0xfd, 0x99, 0xa3, 0xe3, 0x12, 0x56, 0xd1, 0x54, 0xf1, 0xe1,
	       0x24, 0xa0, 0xce, 0xeb, 0x80, 0xb6, 0xde, 0x44, 0xa8, 0xef,
	       0xb1, 0xfc, 0x9c, 0x76, 0xbb, 0x1f, 0x17, 0x40, 0xbe, 0x2a,
	       0xc8, 0x12, 0xa0, 0x82, 0x04, 0x75, 0x30, 0x82, 0x04, 0x71,
	       0x30, 0x82, 0x04, 0x6d, 0x30, 0x82, 0x03, 0x55, 0xa0, 0x03,
	       0x02, 0x01, 0x02, 0x02, 0x10, 0x68, 0x8e, 0x69, 0x1f, 0x05,
	       0x02, 0x11, 0x40, 0x45, 0x8e, 0xf0, 0x39, 0x24, 0x07, 0xcf,
	       0xd6, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
	       0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x81, 0xbc, 0x31,
	       0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
	       0x55, 0x53, 0x31, 0x17, 0x30, 0x15, 0x06, 0x03, 0x55, 0x04,
	       0x0a, 0x13, 0x0e, 0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67,
	       0x6e, 0x2c, 0x20, 0x49, 0x6e, 0x63, 0x2e, 0x31, 0x1f, 0x30,
	       0x1d, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x13, 0x16, 0x56, 0x65,
	       0x72, 0x69, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x54, 0x72, 0x75,
	       0x73, 0x74, 0x20, 0x4e, 0x65, 0x74, 0x77, 0x6f, 0x72, 0x6b,
	       0x31, 0x3b, 0x30, 0x39, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x13,
	       0x32, 0x54, 0x65, 0x72, 0x6d, 0x73, 0x20, 0x6f, 0x66, 0x20,
	       0x75, 0x73, 0x65, 0x20, 0x61, 0x74, 0x20, 0x68, 0x74, 0x74,
	       0x70, 0x73, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x76,
	       0x65, 0x72, 0x69, 0x73, 0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f,
	       0x6d, 0x2f, 0x72, 0x70, 0x61, 0x20, 0x28, 0x63, 0x29, 0x31,
	       0x30, 0x31, 0x36, 0x30, 0x34, 0x06, 0x03, 0x55, 0x04, 0x03,
	       0x13, 0x2d, 0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67, 0x6e,
	       0x20, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x20, 0x33, 0x20, 0x49,
	       0x6e, 0x74, 0x65, 0x72, 0x6e, 0x61, 0x74, 0x69, 0x6f, 0x6e,
	       0x61, 0x6c, 0x20, 0x53, 0x65, 0x72, 0x76, 0x65, 0x72, 0x20,
	       0x43, 0x41, 0x20, 0x2d, 0x20, 0x47, 0x33, 0x30, 0x1e, 0x17,
	       0x0d, 0x31, 0x32, 0x30, 0x33, 0x31, 0x33, 0x30, 0x30, 0x30,
	       0x30, 0x30, 0x30, 0x5a, 0x17, 0x0d, 0x31, 0x32, 0x30, 0x36,
	       0x31, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30,
	       0x81, 0x8e, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04,
	       0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x17, 0x30, 0x15, 0x06,
	       0x03, 0x55, 0x04, 0x0a, 0x13, 0x0e, 0x56, 0x65, 0x72, 0x69,
	       0x53, 0x69, 0x67, 0x6e, 0x2c, 0x20, 0x49, 0x6e, 0x63, 0x2e,
	       0x31, 0x1f, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x13,
	       0x16, 0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67, 0x6e, 0x20,
	       0x54, 0x72, 0x75, 0x73, 0x74, 0x20, 0x4e, 0x65, 0x74, 0x77,
	       0x6f, 0x72, 0x6b, 0x31, 0x45, 0x30, 0x43, 0x06, 0x03, 0x55,
	       0x04, 0x03, 0x13, 0x3c, 0x56, 0x65, 0x72, 0x69, 0x53, 0x69,
	       0x67, 0x6e, 0x20, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x20, 0x33,
	       0x20, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x61, 0x74, 0x69,
	       0x6f, 0x6e, 0x61, 0x6c, 0x20, 0x53, 0x65, 0x72, 0x76, 0x65,
	       0x72, 0x20, 0x43, 0x41, 0x20, 0x2d, 0x20, 0x47, 0x33, 0x20,
	       0x4f, 0x43, 0x53, 0x50, 0x20, 0x52, 0x65, 0x73, 0x70, 0x6f,
	       0x6e, 0x64, 0x65, 0x72, 0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06,
	       0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
	       0x05, 0x00, 0x03, 0x81, 0x8d, 0x00, 0x30, 0x81, 0x89, 0x02,
	       0x81, 0x81, 0x00, 0xa4, 0x3c, 0x3c, 0x43, 0xd3, 0x33, 0x14,
	       0x0a, 0xb4, 0x5e, 0x25, 0x03, 0x6d, 0x60, 0x6c, 0x2f, 0xc9,
	       0x33, 0x11, 0xe6, 0x79, 0xd9, 0x8c, 0x4a, 0x05, 0x60, 0xad,
	       0x16, 0x8b, 0x23, 0x67, 0x8b, 0x1a, 0xaf, 0x84, 0xc3, 0xbb,
	       0x6b, 0xcf, 0x9e, 0xf7, 0x65, 0x6d, 0x04, 0x97, 0xca, 0x12,
	       0x96, 0x0b, 0x30, 0x7f, 0x0d, 0x6e, 0x7f, 0x81, 0x49, 0x53,
	       0xf3, 0xcb, 0x5c, 0x00, 0xd8, 0x6d, 0xf9, 0x03, 0xf4, 0x23,
	       0xd6, 0xd2, 0x9e, 0x8c, 0xca, 0x41, 0xfd, 0xa1, 0x02, 0x20,
	       0xc8, 0x70, 0xb0, 0xfb, 0xaa, 0x1b, 0x33, 0x1e, 0x64, 0x30,
	       0x70, 0x36, 0xed, 0x7b, 0xac, 0xe8, 0x64, 0xd9, 0x79, 0xa8,
	       0x48, 0x20, 0x02, 0xdd, 0x8c, 0x02, 0x6f, 0x87, 0x2f, 0xad,
	       0x29, 0x99, 0x8e, 0x15, 0x2c, 0xd1, 0xe0, 0x64, 0x82, 0x24,
	       0x77, 0x31, 0xee, 0xb8, 0x68, 0xd1, 0x02, 0x32, 0xfb, 0xf0,
	       0xcd, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x82, 0x01, 0x19,
	       0x30, 0x82, 0x01, 0x15, 0x30, 0x09, 0x06, 0x03, 0x55, 0x1d,
	       0x13, 0x04, 0x02, 0x30, 0x00, 0x30, 0x81, 0xac, 0x06, 0x03,
	       0x55, 0x1d, 0x20, 0x04, 0x81, 0xa4, 0x30, 0x81, 0xa1, 0x30,
	       0x81, 0x9e, 0x06, 0x0b, 0x60, 0x86, 0x48, 0x01, 0x86, 0xf8,
	       0x45, 0x01, 0x07, 0x17, 0x03, 0x30, 0x81, 0x8e, 0x30, 0x28,
	       0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x02, 0x01,
	       0x16, 0x1c, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3a, 0x2f, 0x2f,
	       0x77, 0x77, 0x77, 0x2e, 0x76, 0x65, 0x72, 0x69, 0x73, 0x69,
	       0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x43, 0x50, 0x53,
	       0x30, 0x62, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07,
	       0x02, 0x02, 0x30, 0x56, 0x30, 0x15, 0x16, 0x0e, 0x56, 0x65,
	       0x72, 0x69, 0x53, 0x69, 0x67, 0x6e, 0x2c, 0x20, 0x49, 0x6e,
	       0x63, 0x2e, 0x30, 0x03, 0x02, 0x01, 0x01, 0x1a, 0x3d, 0x56,
	       0x65, 0x72, 0x69, 0x53, 0x69, 0x67, 0x6e, 0x27, 0x73, 0x20,
	       0x43, 0x50, 0x53, 0x20, 0x69, 0x6e, 0x63, 0x6f, 0x72, 0x70,
	       0x2e, 0x20, 0x62, 0x79, 0x20, 0x72, 0x65, 0x66, 0x65, 0x72,
	       0x65, 0x6e, 0x63, 0x65, 0x20, 0x6c, 0x69, 0x61, 0x62, 0x2e,
	       0x20, 0x6c, 0x74, 0x64, 0x2e, 0x20, 0x28, 0x63, 0x29, 0x39,
	       0x37, 0x20, 0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67, 0x6e,
	       0x30, 0x13, 0x06, 0x03, 0x55, 0x1d, 0x25, 0x04, 0x0c, 0x30,
	       0x0a, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03,
	       0x09, 0x30, 0x0b, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x04, 0x04,
	       0x03, 0x02, 0x07, 0x80, 0x30, 0x0f, 0x06, 0x09, 0x2b, 0x06,
	       0x01, 0x05, 0x05, 0x07, 0x30, 0x01, 0x05, 0x04, 0x02, 0x05,
	       0x00, 0x30, 0x26, 0x06, 0x03, 0x55, 0x1d, 0x11, 0x04, 0x1f,
	       0x30, 0x1d, 0xa4, 0x1b, 0x30, 0x19, 0x31, 0x17, 0x30, 0x15,
	       0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x0e, 0x4f, 0x43, 0x53,
	       0x50, 0x38, 0x2d, 0x54, 0x47, 0x56, 0x37, 0x2d, 0x32, 0x38,
	       0x35, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
	       0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01,
	       0x00, 0x3b, 0xce, 0xd6, 0xa6, 0x61, 0x41, 0x3d, 0xb2, 0xc4,
	       0x75, 0x00, 0x45, 0xf9, 0x3c, 0x2f, 0x2e, 0x4a, 0x8b, 0xfb,
	       0x9c, 0xff, 0x15, 0xcb, 0x28, 0x65, 0xde, 0xc6, 0x97, 0xd8,
	       0x85, 0x32, 0xa2, 0xfe, 0x74, 0x2d, 0xed, 0x34, 0xf8, 0x3c,
	       0x15, 0xac, 0xfe, 0x72, 0x7f, 0x8c, 0x0e, 0x24, 0xdd, 0xb2,
	       0x80, 0x55, 0xbd, 0x10, 0x0d, 0xa8, 0xb6, 0xf3, 0xe0, 0x98,
	       0x8f, 0x24, 0x0b, 0xa0, 0x9d, 0x2c, 0x05, 0x20, 0x13, 0x4d,
	       0x0b, 0x15, 0x2d, 0x1a, 0x02, 0x76, 0x0a, 0x5a, 0xb2, 0xb9,
	       0xce, 0x6f, 0xb8, 0xc1, 0xec, 0xc4, 0xde, 0x46, 0xe4, 0xec,
	       0x32, 0x05, 0x2d, 0xca, 0xb1, 0x4c, 0x59, 0x41, 0x13, 0x47,
	       0x44, 0x9a, 0x55, 0x20, 0x26, 0xcc, 0x0d, 0x66, 0xb2, 0xd1,
	       0x76, 0x87, 0x74, 0x73, 0xea, 0x5e, 0xe9, 0x49, 0xaf, 0x57,
	       0x9c, 0xa6, 0x3d, 0xe5, 0x2c, 0x28, 0x21, 0xe5, 0x0a, 0x80,
	       0xae, 0x99, 0x44, 0xe9, 0x83, 0x78, 0x7e, 0x24, 0xb5, 0xfa,
	       0x50, 0x9d, 0x32, 0x55, 0x70, 0x73, 0xaf, 0x10, 0x92, 0xd3,
	       0x14, 0xd3, 0x69, 0xfa, 0xa6, 0x81, 0x5c, 0xba, 0x9d, 0x18,
	       0x25, 0xda, 0x78, 0x47, 0x33, 0x83, 0xd1, 0xc3, 0x96, 0x12,
	       0x6f, 0x8b, 0xc9, 0x98, 0x32, 0x79, 0x59, 0x76, 0xcb, 0x56,
	       0x7f, 0x5e, 0x6d, 0x1e, 0x78, 0xb2, 0xd0, 0x39, 0x3c, 0x41,
	       0xd7, 0xab, 0x77, 0x2a, 0x0e, 0xbb, 0xc8, 0x69, 0xae, 0xfd,
	       0x9c, 0x20, 0x11, 0xba, 0x0a, 0x5d, 0x00, 0x58, 0xf8, 0x74,
	       0xb0, 0x77, 0x36, 0x9b, 0x8e, 0xd5, 0x4d, 0x85, 0xb1, 0xbb,
	       0xcd, 0xcb, 0x1d, 0xee, 0x08, 0xb9, 0x53, 0x9a, 0x5f, 0xe3,
	       0x88, 0xb1, 0x1a, 0xc3, 0x41, 0x41, 0x3e, 0x06, 0xf7, 0x70,
	       0x58, 0xe8, 0x22, 0x22, 0xd7, 0x0e, 0xbb, 0x88, 0xce, 0x73,
	       0x0e, 0xd0, 0x04, 0x81, 0x50, 0xd2, 0x33 ) );

OCSP ( google_ocsp, &google_crt, &thawte_crt,
	DATA ( 0x30, 0x51, 0x30, 0x4f, 0x30, 0x4d, 0x30, 0x4b, 0x30, 0x49,
	       0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x05,
	       0x00, 0x04, 0x14, 0x1e, 0x92, 0x09, 0xaa, 0x71, 0x3c, 0x79,
	       0x4b, 0xca, 0x1e, 0x93, 0x1a, 0x0a, 0x61, 0xad, 0x3f, 0xd0,
	       0xba, 0x60, 0x83, 0x04, 0x14, 0x3b, 0x34, 0x9a, 0x70, 0x91,
	       0x73, 0xb2, 0x8a, 0x1b, 0x0c, 0xf4, 0xe9, 0x37, 0xcd, 0xb3,
	       0x70, 0x32, 0x9e, 0x18, 0x54, 0x02, 0x10, 0x4f, 0x9d, 0x96,
	       0xd9, 0x66, 0xb0, 0x99, 0x2b, 0x54, 0xc2, 0x95, 0x7c, 0xb4,
	       0x15, 0x7d, 0x4d ),
	DATA ( 0x30, 0x82, 0x04, 0x38, 0x0a, 0x01, 0x00, 0xa0, 0x82, 0x04,
	       0x31, 0x30, 0x82, 0x04, 0x2d, 0x06, 0x09, 0x2b, 0x06, 0x01,
	       0x05, 0x05, 0x07, 0x30, 0x01, 0x01, 0x04, 0x82, 0x04, 0x1e,
	       0x30, 0x82, 0x04, 0x1a, 0x30, 0x81, 0xe7, 0xa0, 0x03, 0x02,
	       0x01, 0x00, 0xa1, 0x5a, 0x30, 0x58, 0x31, 0x0b, 0x30, 0x09,
	       0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x5a, 0x41, 0x31,
	       0x25, 0x30, 0x23, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x1c,
	       0x54, 0x68, 0x61, 0x77, 0x74, 0x65, 0x20, 0x43, 0x6f, 0x6e,
	       0x73, 0x75, 0x6c, 0x74, 0x69, 0x6e, 0x67, 0x20, 0x28, 0x50,
	       0x74, 0x79, 0x29, 0x20, 0x4c, 0x74, 0x64, 0x2e, 0x31, 0x22,
	       0x30, 0x20, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x19, 0x54,
	       0x68, 0x61, 0x77, 0x74, 0x65, 0x20, 0x53, 0x47, 0x43, 0x20,
	       0x4f, 0x43, 0x53, 0x50, 0x20, 0x52, 0x65, 0x73, 0x70, 0x6f,
	       0x6e, 0x64, 0x65, 0x72, 0x18, 0x0f, 0x32, 0x30, 0x31, 0x32,
	       0x30, 0x35, 0x31, 0x32, 0x31, 0x39, 0x33, 0x31, 0x34, 0x33,
	       0x5a, 0x30, 0x73, 0x30, 0x71, 0x30, 0x49, 0x30, 0x09, 0x06,
	       0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x05, 0x00, 0x04, 0x14,
	       0x1e, 0x92, 0x09, 0xaa, 0x71, 0x3c, 0x79, 0x4b, 0xca, 0x1e,
	       0x93, 0x1a, 0x0a, 0x61, 0xad, 0x3f, 0xd0, 0xba, 0x60, 0x83,
	       0x04, 0x14, 0x3b, 0x34, 0x9a, 0x70, 0x91, 0x73, 0xb2, 0x8a,
	       0x1b, 0x0c, 0xf4, 0xe9, 0x37, 0xcd, 0xb3, 0x70, 0x32, 0x9e,
	       0x18, 0x54, 0x02, 0x10, 0x4f, 0x9d, 0x96, 0xd9, 0x66, 0xb0,
	       0x99, 0x2b, 0x54, 0xc2, 0x95, 0x7c, 0xb4, 0x15, 0x7d, 0x4d,
	       0x80, 0x00, 0x18, 0x0f, 0x32, 0x30, 0x31, 0x32, 0x30, 0x35,
	       0x31, 0x32, 0x31, 0x39, 0x33, 0x31, 0x34, 0x33, 0x5a, 0xa0,
	       0x11, 0x18, 0x0f, 0x32, 0x30, 0x31, 0x32, 0x30, 0x35, 0x31,
	       0x39, 0x31, 0x39, 0x33, 0x31, 0x34, 0x33, 0x5a, 0x30, 0x0d,
	       0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
	       0x05, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00, 0x57, 0xf9, 0xf8,
	       0x26, 0x0f, 0xda, 0x0c, 0xc2, 0xf0, 0xe4, 0x80, 0xd0, 0x29,
	       0xd6, 0x04, 0x77, 0xab, 0x0c, 0xf6, 0x7e, 0x77, 0xc9, 0xac,
	       0x53, 0xa4, 0x4d, 0x6b, 0x75, 0x58, 0xb9, 0xc7, 0x7a, 0x9a,
	       0x39, 0x5d, 0x4c, 0xfd, 0x15, 0x1b, 0x47, 0x55, 0x8e, 0x39,
	       0x05, 0x2b, 0x06, 0x2f, 0x3e, 0x1d, 0x3a, 0x7c, 0x2c, 0xf4,
	       0x4c, 0x70, 0x6a, 0x58, 0x7a, 0x4b, 0x8f, 0x77, 0xaf, 0x9e,
	       0xcf, 0xc6, 0xb1, 0x66, 0x8d, 0xed, 0xbf, 0xc1, 0xd9, 0xe1,
	       0x78, 0x5a, 0x18, 0x61, 0xca, 0xa8, 0xf5, 0x2b, 0xe3, 0x9b,
	       0x50, 0x25, 0x60, 0x7d, 0x8d, 0xd1, 0xe0, 0x76, 0xbb, 0x5c,
	       0x7b, 0xc1, 0x5d, 0xe1, 0x27, 0x3f, 0x7f, 0x9b, 0xb3, 0xb0,
	       0xb2, 0x79, 0x37, 0xd4, 0x53, 0x29, 0x62, 0x46, 0x8f, 0xe6,
	       0x34, 0x9e, 0xe5, 0x6e, 0x63, 0x5d, 0xcc, 0xb3, 0xb2, 0xe2,
	       0x73, 0x7e, 0xa6, 0xad, 0xa0, 0xa0, 0x82, 0x02, 0x99, 0x30,
	       0x82, 0x02, 0x95, 0x30, 0x82, 0x02, 0x91, 0x30, 0x82, 0x01,
	       0xfa, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x67, 0x4d,
	       0x9b, 0x61, 0x8f, 0x57, 0xcd, 0x51, 0xfb, 0x58, 0x07, 0x66,
	       0x1d, 0x24, 0xf5, 0xa5, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86,
	       0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x30,
	       0x4c, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
	       0x13, 0x02, 0x5a, 0x41, 0x31, 0x25, 0x30, 0x23, 0x06, 0x03,
	       0x55, 0x04, 0x0a, 0x13, 0x1c, 0x54, 0x68, 0x61, 0x77, 0x74,
	       0x65, 0x20, 0x43, 0x6f, 0x6e, 0x73, 0x75, 0x6c, 0x74, 0x69,
	       0x6e, 0x67, 0x20, 0x28, 0x50, 0x74, 0x79, 0x29, 0x20, 0x4c,
	       0x74, 0x64, 0x2e, 0x31, 0x16, 0x30, 0x14, 0x06, 0x03, 0x55,
	       0x04, 0x03, 0x13, 0x0d, 0x54, 0x68, 0x61, 0x77, 0x74, 0x65,
	       0x20, 0x53, 0x47, 0x43, 0x20, 0x43, 0x41, 0x30, 0x1e, 0x17,
	       0x0d, 0x31, 0x32, 0x30, 0x33, 0x30, 0x34, 0x30, 0x30, 0x30,
	       0x30, 0x30, 0x30, 0x5a, 0x17, 0x0d, 0x31, 0x32, 0x30, 0x36,
	       0x30, 0x32, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30,
	       0x58, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
	       0x13, 0x02, 0x5a, 0x41, 0x31, 0x25, 0x30, 0x23, 0x06, 0x03,
	       0x55, 0x04, 0x0a, 0x13, 0x1c, 0x54, 0x68, 0x61, 0x77, 0x74,
	       0x65, 0x20, 0x43, 0x6f, 0x6e, 0x73, 0x75, 0x6c, 0x74, 0x69,
	       0x6e, 0x67, 0x20, 0x28, 0x50, 0x74, 0x79, 0x29, 0x20, 0x4c,
	       0x74, 0x64, 0x2e, 0x31, 0x22, 0x30, 0x20, 0x06, 0x03, 0x55,
	       0x04, 0x03, 0x13, 0x19, 0x54, 0x68, 0x61, 0x77, 0x74, 0x65,
	       0x20, 0x53, 0x47, 0x43, 0x20, 0x4f, 0x43, 0x53, 0x50, 0x20,
	       0x52, 0x65, 0x73, 0x70, 0x6f, 0x6e, 0x64, 0x65, 0x72, 0x30,
	       0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
	       0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8d,
	       0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xc2, 0x21,
	       0x52, 0xe2, 0x10, 0x44, 0xe3, 0xc7, 0xe3, 0x72, 0x19, 0x01,
	       0xa8, 0x0c, 0x5a, 0x8b, 0x63, 0x0b, 0x30, 0x4c, 0x4f, 0xa2,
	       0xe2, 0xdf, 0x62, 0x2e, 0xdf, 0xe2, 0x77, 0xb9, 0x27, 0x3b,
	       0x8d, 0x45, 0xe2, 0xa1, 0x21, 0x95, 0x13, 0x5a, 0xf2, 0x7d,
	       0x58, 0x2c, 0x6d, 0x53, 0xb2, 0x42, 0x52, 0x8f, 0x8e, 0xf4,
	       0x17, 0xc5, 0xcb, 0xfa, 0x2f, 0x2d, 0x9b, 0x02, 0xd0, 0x1a,
	       0xf1, 0x32, 0xc5, 0xb9, 0xd3, 0x5a, 0xc0, 0xd6, 0x20, 0xb8,
	       0x3f, 0xa3, 0x93, 0x8b, 0xe5, 0x22, 0x91, 0x1a, 0x7e, 0x7c,
	       0x8b, 0x35, 0xeb, 0x94, 0xee, 0xe6, 0xcc, 0x58, 0xe4, 0x3c,
	       0xcd, 0x00, 0x86, 0xaa, 0xc3, 0x2e, 0x8b, 0xb4, 0x3c, 0x29,
	       0x11, 0x21, 0x5d, 0x71, 0x4a, 0xb6, 0x7a, 0x22, 0xa9, 0xf2,
	       0xf8, 0x90, 0xed, 0x5c, 0x73, 0x8c, 0xe0, 0x70, 0x56, 0xde,
	       0x70, 0xc5, 0x0d, 0x81, 0xb4, 0x5b, 0x02, 0x03, 0x01, 0x00,
	       0x01, 0xa3, 0x68, 0x30, 0x66, 0x30, 0x09, 0x06, 0x03, 0x55,
	       0x1d, 0x13, 0x04, 0x02, 0x30, 0x00, 0x30, 0x13, 0x06, 0x03,
	       0x55, 0x1d, 0x25, 0x04, 0x0c, 0x30, 0x0a, 0x06, 0x08, 0x2b,
	       0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x09, 0x30, 0x0b, 0x06,
	       0x03, 0x55, 0x1d, 0x0f, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80,
	       0x30, 0x0f, 0x06, 0x09, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07,
	       0x30, 0x01, 0x05, 0x04, 0x02, 0x05, 0x00, 0x30, 0x26, 0x06,
	       0x03, 0x55, 0x1d, 0x11, 0x04, 0x1f, 0x30, 0x1d, 0xa4, 0x1b,
	       0x30, 0x19, 0x31, 0x17, 0x30, 0x15, 0x06, 0x03, 0x55, 0x04,
	       0x03, 0x13, 0x0e, 0x4f, 0x43, 0x53, 0x50, 0x38, 0x2d, 0x54,
	       0x47, 0x56, 0x37, 0x2d, 0x32, 0x37, 0x35, 0x30, 0x0d, 0x06,
	       0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05,
	       0x05, 0x00, 0x03, 0x81, 0x81, 0x00, 0x98, 0x50, 0x14, 0xbe,
	       0x31, 0x44, 0x01, 0x97, 0xee, 0x21, 0xf9, 0xcf, 0xa4, 0x32,
	       0xc2, 0x91, 0x6a, 0x4b, 0x02, 0x61, 0x62, 0xa9, 0xe4, 0xad,
	       0x8a, 0xac, 0xe4, 0x12, 0x12, 0xbe, 0x9b, 0xba, 0x4c, 0xe2,
	       0x92, 0xdb, 0x8b, 0x92, 0x11, 0xe6, 0x9e, 0x25, 0x2e, 0xd5,
	       0x4b, 0xe8, 0x5d, 0xc8, 0x63, 0xd2, 0xff, 0xfa, 0x80, 0x89,
	       0x83, 0x0f, 0xe4, 0xe2, 0x91, 0xcc, 0x0a, 0x85, 0x74, 0x38,
	       0xb7, 0xdf, 0xb9, 0x1c, 0xf6, 0x35, 0xfe, 0x6f, 0xf3, 0x29,
	       0xc0, 0xc5, 0x0a, 0x2c, 0x04, 0xd7, 0x69, 0xff, 0x02, 0x45,
	       0x1c, 0x29, 0x7a, 0x27, 0xf8, 0xb2, 0xe3, 0x58, 0x09, 0x34,
	       0xc4, 0xd7, 0x77, 0x74, 0xbd, 0xe4, 0x7c, 0xda, 0x99, 0x09,
	       0x83, 0x03, 0x6c, 0x36, 0x0d, 0xb2, 0x91, 0x71, 0x40, 0xc7,
	       0x97, 0x85, 0xfb, 0x2a, 0xa3, 0x92, 0x65, 0x0b, 0x02, 0x58,
	       0x14, 0x89, 0x8f, 0x3b ) );

OCSP ( unauthorized_ocsp, &barclays_crt, &thawte_crt,
	DATA ( 0x30, 0x51, 0x30, 0x4f, 0x30, 0x4d, 0x30, 0x4b, 0x30, 0x49,
	       0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x05,
	       0x00, 0x04, 0x14, 0xd4, 0xb4, 0x3b, 0x8e, 0x3d, 0x02, 0x49,
	       0x1a, 0x65, 0x50, 0x6f, 0x96, 0x73, 0x14, 0xdd, 0xe8, 0x59,
	       0x44, 0x52, 0xe4, 0x04, 0x14, 0x3b, 0x34, 0x9a, 0x70, 0x91,
	       0x73, 0xb2, 0x8a, 0x1b, 0x0c, 0xf4, 0xe9, 0x37, 0xcd, 0xb3,
	       0x70, 0x32, 0x9e, 0x18, 0x54, 0x02, 0x10, 0x49, 0x83, 0xfc,
	       0x05, 0x76, 0xdf, 0x36, 0x91, 0x7c, 0x64, 0x2a, 0x27, 0xc1,
	       0xf1, 0x48, 0xe3 ),
	DATA ( 0x30, 0x03, 0x0a, 0x01, 0x06 ) );

OCSP ( unknown_ocsp, &thawte_crt, &startssl_crt,
	DATA ( 0x30, 0x45, 0x30, 0x43, 0x30, 0x41, 0x30, 0x3f, 0x30, 0x3d,
	       0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x05,
	       0x00, 0x04, 0x14, 0x48, 0xb7, 0x64, 0x49, 0xf3, 0xd5, 0xfe,
	       0xfa, 0x11, 0x33, 0xaa, 0x80, 0x5e, 0x42, 0x0f, 0x0f, 0xca,
	       0x64, 0x36, 0x51, 0x04, 0x14, 0x11, 0xdb, 0x23, 0x45, 0xfd,
	       0x54, 0xcc, 0x6a, 0x71, 0x6f, 0x84, 0x8a, 0x03, 0xd7, 0xbe,
	       0xf7, 0x01, 0x2f, 0x26, 0x86, 0x02, 0x04, 0x30, 0x00, 0x00,
	       0x02 ),
	DATA ( 0x30, 0x82, 0x06, 0x46, 0x0a, 0x01, 0x00, 0xa0, 0x82, 0x06,
	       0x3f, 0x30, 0x82, 0x06, 0x3b, 0x06, 0x09, 0x2b, 0x06, 0x01,
	       0x05, 0x05, 0x07, 0x30, 0x01, 0x01, 0x04, 0x82, 0x06, 0x2c,
	       0x30, 0x82, 0x06, 0x28, 0x30, 0x81, 0xec, 0xa1, 0x70, 0x30,
	       0x6e, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
	       0x13, 0x02, 0x49, 0x4c, 0x31, 0x31, 0x30, 0x2f, 0x06, 0x03,
	       0x55, 0x04, 0x0a, 0x13, 0x28, 0x53, 0x74, 0x61, 0x72, 0x74,
	       0x43, 0x6f, 0x6d, 0x20, 0x4c, 0x74, 0x64, 0x2e, 0x20, 0x28,
	       0x53, 0x74, 0x61, 0x72, 0x74, 0x20, 0x43, 0x6f, 0x6d, 0x6d,
	       0x65, 0x72, 0x63, 0x69, 0x61, 0x6c, 0x20, 0x4c, 0x69, 0x6d,
	       0x69, 0x74, 0x65, 0x64, 0x29, 0x31, 0x2c, 0x30, 0x2a, 0x06,
	       0x03, 0x55, 0x04, 0x03, 0x13, 0x23, 0x53, 0x74, 0x61, 0x72,
	       0x74, 0x43, 0x6f, 0x6d, 0x20, 0x43, 0x6c, 0x61, 0x73, 0x73,
	       0x20, 0x32, 0x20, 0x53, 0x65, 0x72, 0x76, 0x65, 0x72, 0x20,
	       0x4f, 0x43, 0x53, 0x50, 0x20, 0x53, 0x69, 0x67, 0x6e, 0x65,
	       0x72, 0x18, 0x0f, 0x32, 0x30, 0x31, 0x32, 0x30, 0x35, 0x31,
	       0x35, 0x31, 0x31, 0x35, 0x34, 0x30, 0x38, 0x5a, 0x30, 0x67,
	       0x30, 0x65, 0x30, 0x3d, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e,
	       0x03, 0x02, 0x1a, 0x05, 0x00, 0x04, 0x14, 0x48, 0xb7, 0x64,
	       0x49, 0xf3, 0xd5, 0xfe, 0xfa, 0x11, 0x33, 0xaa, 0x80, 0x5e,
	       0x42, 0x0f, 0x0f, 0xca, 0x64, 0x36, 0x51, 0x04, 0x14, 0x11,
	       0xdb, 0x23, 0x45, 0xfd, 0x54, 0xcc, 0x6a, 0x71, 0x6f, 0x84,
	       0x8a, 0x03, 0xd7, 0xbe, 0xf7, 0x01, 0x2f, 0x26, 0x86, 0x02,
	       0x04, 0x30, 0x00, 0x00, 0x02, 0x82, 0x00, 0x18, 0x0f, 0x32,
	       0x30, 0x31, 0x32, 0x30, 0x35, 0x31, 0x35, 0x31, 0x31, 0x35,
	       0x34, 0x30, 0x38, 0x5a, 0xa0, 0x11, 0x18, 0x0f, 0x32, 0x30,
	       0x31, 0x32, 0x30, 0x35, 0x31, 0x36, 0x31, 0x31, 0x35, 0x34,
	       0x30, 0x38, 0x5a, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48,
	       0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x82,
	       0x01, 0x01, 0x00, 0xbe, 0x1b, 0xa7, 0xa7, 0xba, 0x17, 0xd9,
	       0x94, 0x16, 0xdf, 0xd2, 0x86, 0x1e, 0x39, 0x38, 0x2f, 0x0e,
	       0x0e, 0xb0, 0x19, 0x74, 0x9d, 0x64, 0x61, 0xfb, 0x34, 0x15,
	       0x64, 0xe6, 0x7a, 0x44, 0xfc, 0x24, 0xaf, 0x63, 0xe3, 0xe5,
	       0x01, 0x3f, 0xeb, 0x62, 0xc4, 0x2f, 0xd7, 0x56, 0xac, 0x9e,
	       0x39, 0x8c, 0x54, 0x20, 0x24, 0x9f, 0xe0, 0x9a, 0x2c, 0x9a,
	       0xfb, 0xbe, 0x13, 0x8d, 0x18, 0xf1, 0x95, 0x37, 0xf7, 0x6a,
	       0x93, 0x28, 0x2a, 0xf6, 0x10, 0xc0, 0x5e, 0xa0, 0xfc, 0xf7,
	       0x66, 0x97, 0xe4, 0x76, 0x04, 0x90, 0xd3, 0x45, 0x59, 0x26,
	       0xfd, 0xe9, 0xb4, 0xe5, 0xd6, 0x30, 0x2f, 0xe0, 0xfb, 0xda,
	       0xcc, 0x4b, 0xc4, 0x11, 0xbf, 0x20, 0x50, 0x18, 0xd5, 0x18,
	       0xfc, 0xe7, 0x86, 0xb8, 0x80, 0x2b, 0x2e, 0x35, 0x50, 0xcd,
	       0x73, 0x0d, 0x70, 0xbe, 0x55, 0xa2, 0xef, 0x2c, 0x62, 0x96,
	       0xe3, 0x6a, 0xec, 0x69, 0xa6, 0x8f, 0x9d, 0x37, 0xb6, 0xbe,
	       0x6b, 0x72, 0x02, 0x99, 0x02, 0xea, 0x0b, 0x18, 0x01, 0x26,
	       0x82, 0x3b, 0x7b, 0x44, 0x8a, 0x84, 0xe4, 0x78, 0x6c, 0xb3,
	       0x5b, 0x83, 0x87, 0x7c, 0xab, 0x80, 0x17, 0xfd, 0x00, 0xfd,
	       0x56, 0x87, 0x85, 0x2b, 0x49, 0x42, 0xa2, 0x63, 0x84, 0x4f,
	       0x4a, 0xaa, 0x5e, 0x7d, 0x64, 0x29, 0x09, 0x81, 0xac, 0xea,
	       0x53, 0x00, 0x36, 0xbf, 0x19, 0x33, 0x5c, 0x0e, 0xee, 0xa9,
	       0x6a, 0x9e, 0x2e, 0x44, 0x9b, 0x3e, 0xc9, 0x27, 0xb7, 0x49,
	       0x15, 0x76, 0xa8, 0x42, 0x79, 0x2d, 0x4a, 0x42, 0x1f, 0xf1,
	       0x32, 0x35, 0x31, 0x4c, 0xcb, 0xa2, 0xee, 0x50, 0xae, 0x1f,
	       0x5e, 0x4d, 0x5d, 0xc1, 0x9e, 0x28, 0x17, 0x59, 0x6d, 0x1d,
	       0x2a, 0x0a, 0x3d, 0xeb, 0x88, 0x10, 0xb5, 0xe6, 0x3b, 0xa7,
	       0x81, 0x1f, 0xc7, 0xa0, 0xc8, 0x4a, 0xfe, 0x6c, 0x8f, 0xa0,
	       0x82, 0x04, 0x21, 0x30, 0x82, 0x04, 0x1d, 0x30, 0x82, 0x04,
	       0x19, 0x30, 0x82, 0x03, 0x01, 0xa0, 0x03, 0x02, 0x01, 0x02,
	       0x02, 0x03, 0x00, 0xca, 0x67, 0x30, 0x0d, 0x06, 0x09, 0x2a,
	       0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05, 0x00,
	       0x30, 0x81, 0x8c, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55,
	       0x04, 0x06, 0x13, 0x02, 0x49, 0x4c, 0x31, 0x16, 0x30, 0x14,
	       0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x0d, 0x53, 0x74, 0x61,
	       0x72, 0x74, 0x43, 0x6f, 0x6d, 0x20, 0x4c, 0x74, 0x64, 0x2e,
	       0x31, 0x2b, 0x30, 0x29, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x13,
	       0x22, 0x53, 0x65, 0x63, 0x75, 0x72, 0x65, 0x20, 0x44, 0x69,
	       0x67, 0x69, 0x74, 0x61, 0x6c, 0x20, 0x43, 0x65, 0x72, 0x74,
	       0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x53, 0x69,
	       0x67, 0x6e, 0x69, 0x6e, 0x67, 0x31, 0x38, 0x30, 0x36, 0x06,
	       0x03, 0x55, 0x04, 0x03, 0x13, 0x2f, 0x53, 0x74, 0x61, 0x72,
	       0x74, 0x43, 0x6f, 0x6d, 0x20, 0x43, 0x6c, 0x61, 0x73, 0x73,
	       0x20, 0x32, 0x20, 0x50, 0x72, 0x69, 0x6d, 0x61, 0x72, 0x79,
	       0x20, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6d, 0x65, 0x64, 0x69,
	       0x61, 0x74, 0x65, 0x20, 0x53, 0x65, 0x72, 0x76, 0x65, 0x72,
	       0x20, 0x43, 0x41, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x32, 0x30,
	       0x34, 0x32, 0x32, 0x30, 0x35, 0x33, 0x30, 0x35, 0x39, 0x5a,
	       0x17, 0x0d, 0x31, 0x32, 0x30, 0x36, 0x30, 0x32, 0x30, 0x36,
	       0x32, 0x39, 0x32, 0x37, 0x5a, 0x30, 0x6e, 0x31, 0x0b, 0x30,
	       0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x49, 0x4c,
	       0x31, 0x31, 0x30, 0x2f, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13,
	       0x28, 0x53, 0x74, 0x61, 0x72, 0x74, 0x43, 0x6f, 0x6d, 0x20,
	       0x4c, 0x74, 0x64, 0x2e, 0x20, 0x28, 0x53, 0x74, 0x61, 0x72,
	       0x74, 0x20, 0x43, 0x6f, 0x6d, 0x6d, 0x65, 0x72, 0x63, 0x69,
	       0x61, 0x6c, 0x20, 0x4c, 0x69, 0x6d, 0x69, 0x74, 0x65, 0x64,
	       0x29, 0x31, 0x2c, 0x30, 0x2a, 0x06, 0x03, 0x55, 0x04, 0x03,
	       0x13, 0x23, 0x53, 0x74, 0x61, 0x72, 0x74, 0x43, 0x6f, 0x6d,
	       0x20, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x20, 0x32, 0x20, 0x53,
	       0x65, 0x72, 0x76, 0x65, 0x72, 0x20, 0x4f, 0x43, 0x53, 0x50,
	       0x20, 0x53, 0x69, 0x67, 0x6e, 0x65, 0x72, 0x30, 0x82, 0x01,
	       0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
	       0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f,
	       0x00, 0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00,
	       0xd0, 0xd5, 0xf5, 0x70, 0x9a, 0x2b, 0x41, 0x35, 0xf7, 0xf8,
	       0xcf, 0xb5, 0x6b, 0xb3, 0xdf, 0xe0, 0xb9, 0x12, 0x0d, 0x3d,
	       0xd3, 0x99, 0x9f, 0x32, 0x73, 0x01, 0x1f, 0xbc, 0x7d, 0x4f,
	       0x3e, 0x66, 0xf7, 0xfd, 0x60, 0x57, 0x92, 0x30, 0xb4, 0xdb,
	       0x9a, 0xf5, 0xd3, 0x49, 0x19, 0xd6, 0xad, 0x37, 0x43, 0x78,
	       0x69, 0x8c, 0x0d, 0x23, 0x0e, 0x7a, 0xd1, 0x65, 0x08, 0xeb,
	       0x71, 0x8c, 0x37, 0x36, 0xd3, 0x4d, 0xa6, 0xcb, 0x11, 0xb2,
	       0xfa, 0xb4, 0x38, 0x3e, 0x2b, 0x70, 0x8c, 0xf7, 0xf1, 0xd9,
	       0x64, 0x62, 0x26, 0xf4, 0xa7, 0x2c, 0x24, 0x25, 0x4e, 0x4d,
	       0x3e, 0x7a, 0x54, 0x57, 0x0f, 0xc1, 0x89, 0x9e, 0xb6, 0x55,
	       0x0b, 0x7c, 0xbe, 0x38, 0xda, 0x8b, 0x62, 0xe9, 0xf1, 0xfa,
	       0x8c, 0xd9, 0x32, 0x1f, 0xbe, 0x6d, 0x2e, 0x3d, 0x48, 0xa7,
	       0x4f, 0x48, 0xd4, 0xff, 0x6b, 0xf6, 0x17, 0xf8, 0x31, 0xb2,
	       0x37, 0xeb, 0x89, 0x71, 0x19, 0x0f, 0xe7, 0x86, 0x06, 0x66,
	       0xfb, 0xc5, 0xad, 0x7b, 0x75, 0x0b, 0xcc, 0x2e, 0x3c, 0x4d,
	       0x1c, 0x99, 0x40, 0x32, 0x72, 0xd4, 0x5c, 0xc9, 0x06, 0xaa,
	       0x98, 0xe9, 0x01, 0x92, 0xdb, 0x25, 0x48, 0x1a, 0xae, 0x3f,
	       0x01, 0x4d, 0x8a, 0xb0, 0x78, 0xb1, 0x28, 0xe0, 0x09, 0x9b,
	       0x23, 0xe2, 0x28, 0x46, 0x6f, 0x50, 0x52, 0x71, 0x1c, 0xf1,
	       0x09, 0xa0, 0x87, 0x3b, 0xdb, 0x84, 0xa3, 0xb1, 0x57, 0x6f,
	       0xbf, 0x52, 0xd2, 0x30, 0x83, 0x30, 0x26, 0xc0, 0x27, 0x8e,
	       0x6d, 0x03, 0x43, 0x14, 0x42, 0x31, 0x29, 0xf2, 0x7e, 0x52,
	       0xcb, 0x84, 0x20, 0x2e, 0x87, 0x19, 0xe5, 0x48, 0xad, 0x06,
	       0xce, 0x2e, 0x0f, 0xed, 0x78, 0x2a, 0x3d, 0x79, 0xc4, 0xb0,
	       0xdb, 0xfa, 0x4e, 0x95, 0x88, 0x46, 0x75, 0x12, 0xb0, 0x7a,
	       0x55, 0x6a, 0x38, 0xae, 0xea, 0x59, 0x02, 0x03, 0x01, 0x00,
	       0x01, 0xa3, 0x81, 0xa0, 0x30, 0x81, 0x9d, 0x30, 0x09, 0x06,
	       0x03, 0x55, 0x1d, 0x13, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0b,
	       0x06, 0x03, 0x55, 0x1d, 0x0f, 0x04, 0x04, 0x03, 0x02, 0x03,
	       0xa8, 0x30, 0x1e, 0x06, 0x03, 0x55, 0x1d, 0x25, 0x04, 0x17,
	       0x30, 0x15, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07,
	       0x03, 0x09, 0x06, 0x09, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07,
	       0x30, 0x01, 0x05, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e,
	       0x04, 0x16, 0x04, 0x14, 0xbd, 0x4c, 0xef, 0x0e, 0xf7, 0x08,
	       0xac, 0xc9, 0xbd, 0x39, 0x0f, 0xd9, 0xa0, 0xd3, 0xce, 0xcf,
	       0x26, 0x48, 0xb8, 0x19, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d,
	       0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x11, 0xdb, 0x23,
	       0x45, 0xfd, 0x54, 0xcc, 0x6a, 0x71, 0x6f, 0x84, 0x8a, 0x03,
	       0xd7, 0xbe, 0xf7, 0x01, 0x2f, 0x26, 0x86, 0x30, 0x23, 0x06,
	       0x03, 0x55, 0x1d, 0x12, 0x04, 0x1c, 0x30, 0x1a, 0x86, 0x18,
	       0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77,
	       0x2e, 0x73, 0x74, 0x61, 0x72, 0x74, 0x73, 0x73, 0x6c, 0x2e,
	       0x63, 0x6f, 0x6d, 0x2f, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86,
	       0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03,
	       0x82, 0x01, 0x01, 0x00, 0x56, 0x1f, 0xef, 0xa5, 0x1a, 0x07,
	       0xbe, 0xb1, 0xd1, 0xd0, 0x17, 0xeb, 0x72, 0x5b, 0x17, 0x11,
	       0xe5, 0x96, 0xc3, 0x96, 0xe7, 0x9a, 0xdb, 0xbf, 0x64, 0x1c,
	       0x99, 0x11, 0x2f, 0x18, 0x07, 0xb2, 0x45, 0x48, 0xf3, 0x58,
	       0xcd, 0x38, 0x69, 0x33, 0xf4, 0x58, 0x5b, 0x16, 0xed, 0xfb,
	       0xce, 0xb3, 0xc3, 0x14, 0x27, 0xa6, 0x16, 0xff, 0xd0, 0x70,
	       0x9e, 0xe1, 0x9d, 0x4b, 0xd1, 0x26, 0x6c, 0x61, 0x25, 0xf1,
	       0x39, 0x9c, 0xbe, 0x69, 0x75, 0x58, 0xcd, 0xbd, 0x8e, 0x36,
	       0xfd, 0x46, 0xd1, 0xe3, 0xb9, 0x1a, 0x8a, 0xc1, 0xd7, 0x3e,
	       0x6e, 0x82, 0xb8, 0xb0, 0x3f, 0xcf, 0x14, 0x3f, 0xc6, 0xf6,
	       0x3a, 0x86, 0xce, 0x03, 0x76, 0x1f, 0xdb, 0x0b, 0x12, 0xac,
	       0x99, 0x79, 0x53, 0xf0, 0x3d, 0x70, 0xd3, 0x5a, 0x05, 0xf6,
	       0xba, 0x6e, 0x35, 0x31, 0x1e, 0x08, 0x30, 0xc1, 0xa4, 0xd4,
	       0x45, 0x43, 0x5a, 0x01, 0xd9, 0x3d, 0xa5, 0xdb, 0xd2, 0xd7,
	       0x73, 0x97, 0xe9, 0xab, 0xe4, 0x60, 0xf1, 0xfc, 0xf0, 0x9b,
	       0xe2, 0x5a, 0x1e, 0x31, 0xe0, 0x1b, 0x47, 0x3f, 0x5a, 0x78,
	       0xf3, 0x6e, 0xf0, 0x94, 0x6c, 0x2c, 0xfb, 0x67, 0x6e, 0xcb,
	       0x8c, 0xb6, 0x8d, 0xcc, 0xcf, 0x1e, 0x9f, 0xd2, 0x10, 0x52,
	       0xc2, 0xe7, 0xc8, 0x05, 0x2c, 0xa0, 0x18, 0xf5, 0x53, 0x4a,
	       0xd2, 0xb0, 0x57, 0x5e, 0x5f, 0x63, 0xd7, 0x7b, 0x8e, 0xfa,
	       0x22, 0xa0, 0x69, 0x17, 0xd2, 0xa0, 0xc7, 0x70, 0x01, 0x79,
	       0x8b, 0x69, 0x1f, 0x0f, 0xdb, 0xe5, 0xf9, 0x83, 0x2b, 0x26,
	       0x05, 0x05, 0x87, 0x80, 0x0d, 0xf9, 0x20, 0x0e, 0x16, 0x39,
	       0xc5, 0x9b, 0x14, 0x2e, 0xf2, 0x06, 0x57, 0x46, 0x3d, 0x0b,
	       0x8c, 0x3e, 0xb4, 0x66, 0x76, 0x67, 0x34, 0x70, 0x00, 0x63,
	       0xcf, 0x9e, 0xc8, 0xc5, 0x5f, 0x48, 0x06, 0x53, 0x26, 0x55 ) );

OCSP ( vultr_ocsp, &vultr_crt, &rapidssl_crt,
	DATA ( 0x30, 0x44, 0x30, 0x42, 0x30, 0x40, 0x30, 0x3e, 0x30, 0x3c,
	       0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x05,
	       0x00, 0x04, 0x14, 0x40, 0x0b, 0x46, 0x7a, 0xf1, 0xe6, 0xb2,
	       0xd3, 0x09, 0x83, 0xba, 0x0d, 0x60, 0x7e, 0x7e, 0x59, 0x37,
	       0x48, 0x24, 0xc4, 0x04, 0x14, 0xc3, 0x9c, 0xf3, 0xfc, 0xd3,
	       0x46, 0x08, 0x34, 0xbb, 0xce, 0x46, 0x7f, 0xa0, 0x7c, 0x5b,
	       0xf3, 0xe2, 0x08, 0xcb, 0x59, 0x02, 0x03, 0x00, 0x95, 0x4d ),
	DATA ( 0x30, 0x82, 0x05, 0x70, 0x0a, 0x01, 0x00, 0xa0, 0x82, 0x05,
	       0x69, 0x30, 0x82, 0x05, 0x65, 0x06, 0x09, 0x2b, 0x06, 0x01,
	       0x05, 0x05, 0x07, 0x30, 0x01, 0x01, 0x04, 0x82, 0x05, 0x56,
	       0x30, 0x82, 0x05, 0x52, 0x30, 0x81, 0x91, 0xa2, 0x16, 0x04,
	       0x14, 0xfa, 0x58, 0xdb, 0x09, 0x53, 0xbc, 0x19, 0xc5, 0xe7,
	       0xb5, 0x8b, 0xf6, 0x10, 0xf8, 0x1e, 0x84, 0x6d, 0x3a, 0x8f,
	       0xd8, 0x18, 0x0f, 0x32, 0x30, 0x31, 0x34, 0x31, 0x31, 0x32,
	       0x32, 0x32, 0x33, 0x30, 0x38, 0x35, 0x36, 0x5a, 0x30, 0x66,
	       0x30, 0x64, 0x30, 0x3c, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e,
	       0x03, 0x02, 0x1a, 0x05, 0x00, 0x04, 0x14, 0x40, 0x0b, 0x46,
	       0x7a, 0xf1, 0xe6, 0xb2, 0xd3, 0x09, 0x83, 0xba, 0x0d, 0x60,
	       0x7e, 0x7e, 0x59, 0x37, 0x48, 0x24, 0xc4, 0x04, 0x14, 0xc3,
	       0x9c, 0xf3, 0xfc, 0xd3, 0x46, 0x08, 0x34, 0xbb, 0xce, 0x46,
	       0x7f, 0xa0, 0x7c, 0x5b, 0xf3, 0xe2, 0x08, 0xcb, 0x59, 0x02,
	       0x03, 0x00, 0x95, 0x4d, 0x80, 0x00, 0x18, 0x0f, 0x32, 0x30,
	       0x31, 0x34, 0x31, 0x31, 0x32, 0x32, 0x32, 0x33, 0x30, 0x38,
	       0x35, 0x36, 0x5a, 0xa0, 0x11, 0x18, 0x0f, 0x32, 0x30, 0x31,
	       0x34, 0x31, 0x31, 0x32, 0x39, 0x32, 0x33, 0x30, 0x38, 0x35,
	       0x36, 0x5a, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
	       0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x82, 0x01,
	       0x01, 0x00, 0x6a, 0x71, 0x8f, 0x84, 0x66, 0xb5, 0x75, 0xe6,
	       0x97, 0xa4, 0xb9, 0xc6, 0xa0, 0x37, 0x6f, 0x23, 0x76, 0x3c,
	       0x59, 0x4c, 0x1c, 0x2d, 0x9f, 0x70, 0xab, 0x83, 0xbf, 0xa9,
	       0xbf, 0x79, 0x31, 0x69, 0xdd, 0x78, 0xd5, 0x59, 0x90, 0x68,
	       0xbe, 0x25, 0xb7, 0x53, 0x7d, 0x8b, 0xcf, 0x66, 0x3b, 0xcd,
	       0xe0, 0xd2, 0x40, 0x1d, 0xc8, 0x29, 0xe4, 0x37, 0xbf, 0x20,
	       0x7e, 0x64, 0x8d, 0x0d, 0xc7, 0xed, 0x0d, 0x08, 0x05, 0x36,
	       0x27, 0x4f, 0xb8, 0xe3, 0x19, 0xec, 0xf0, 0x96, 0xe8, 0x48,
	       0x9b, 0x8b, 0x2c, 0x18, 0xdb, 0x1e, 0x68, 0x11, 0xf3, 0xfb,
	       0x9c, 0x68, 0xad, 0xcc, 0x15, 0xe0, 0x25, 0x08, 0x98, 0xd2,
	       0xbf, 0xd0, 0x57, 0xe6, 0x4c, 0x73, 0x5a, 0x2c, 0xc8, 0x89,
	       0xd6, 0xe4, 0xd0, 0x47, 0x6d, 0x8c, 0xc7, 0x75, 0xb1, 0x4e,
	       0x10, 0x34, 0xe5, 0x40, 0xa3, 0xb1, 0x50, 0x07, 0x3d, 0x7d,
	       0xad, 0xeb, 0x1d, 0x91, 0x7f, 0x77, 0x2e, 0x0d, 0x9a, 0xa7,
	       0xbb, 0x68, 0x89, 0xd2, 0x05, 0x58, 0x16, 0xf1, 0x5e, 0x1d,
	       0x05, 0xf6, 0x9e, 0xe9, 0x89, 0x52, 0x35, 0xb7, 0x29, 0x7a,
	       0x68, 0x02, 0x6f, 0xc7, 0x20, 0x30, 0xc8, 0xde, 0x97, 0x3f,
	       0xb7, 0x28, 0x38, 0x39, 0xd1, 0x4b, 0x4b, 0x90, 0x71, 0xe5,
	       0x58, 0xa4, 0xa3, 0xbd, 0x78, 0x95, 0xb5, 0x54, 0xdd, 0xf7,
	       0x4f, 0x8e, 0x78, 0x73, 0x86, 0xbf, 0x28, 0xb0, 0xdd, 0xc0,
	       0xe9, 0x4a, 0xf5, 0x9f, 0x02, 0x8e, 0x63, 0x8f, 0x59, 0xf1,
	       0x93, 0xf0, 0x45, 0x97, 0x30, 0xdb, 0x0a, 0x04, 0x3e, 0x81,
	       0x99, 0x20, 0x7a, 0xb2, 0xe6, 0x8c, 0x8f, 0x2a, 0x4c, 0x31,
	       0xf1, 0x64, 0xbc, 0xb7, 0xec, 0xb1, 0xf9, 0x69, 0x1f, 0x99,
	       0x89, 0x3e, 0x3e, 0xa0, 0xf4, 0xde, 0x79, 0xa7, 0xae, 0xa3,
	       0x23, 0xbd, 0x16, 0xbb, 0x6d, 0x0f, 0x15, 0x68, 0xa0, 0x82,
	       0x03, 0xa6, 0x30, 0x82, 0x03, 0xa2, 0x30, 0x82, 0x03, 0x9e,
	       0x30, 0x82, 0x02, 0x86, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02,
	       0x01, 0x04, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
	       0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x30, 0x47, 0x31,
	       0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
	       0x55, 0x53, 0x31, 0x16, 0x30, 0x14, 0x06, 0x03, 0x55, 0x04,
	       0x0a, 0x13, 0x0d, 0x47, 0x65, 0x6f, 0x54, 0x72, 0x75, 0x73,
	       0x74, 0x20, 0x49, 0x6e, 0x63, 0x2e, 0x31, 0x20, 0x30, 0x1e,
	       0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x17, 0x52, 0x61, 0x70,
	       0x69, 0x64, 0x53, 0x53, 0x4c, 0x20, 0x53, 0x48, 0x41, 0x32,
	       0x35, 0x36, 0x20, 0x43, 0x41, 0x20, 0x2d, 0x20, 0x47, 0x33,
	       0x30, 0x1e, 0x17, 0x0d, 0x31, 0x34, 0x30, 0x38, 0x32, 0x39,
	       0x32, 0x33, 0x33, 0x39, 0x33, 0x30, 0x5a, 0x17, 0x0d, 0x31,
	       0x35, 0x30, 0x35, 0x32, 0x32, 0x32, 0x33, 0x33, 0x39, 0x33,
	       0x30, 0x5a, 0x30, 0x31, 0x31, 0x2f, 0x30, 0x2d, 0x06, 0x03,
	       0x55, 0x04, 0x03, 0x13, 0x26, 0x52, 0x61, 0x70, 0x69, 0x64,
	       0x53, 0x53, 0x4c, 0x20, 0x53, 0x48, 0x41, 0x32, 0x35, 0x36,
	       0x20, 0x43, 0x41, 0x20, 0x2d, 0x20, 0x47, 0x33, 0x20, 0x4f,
	       0x43, 0x53, 0x50, 0x20, 0x52, 0x65, 0x73, 0x70, 0x6f, 0x6e,
	       0x64, 0x65, 0x72, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0d, 0x06,
	       0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
	       0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30, 0x82, 0x01,
	       0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0x9b, 0xf2, 0x8e, 0xe9,
	       0x57, 0x3e, 0xa8, 0x5c, 0xfd, 0x00, 0x14, 0x21, 0xe7, 0xe4,
	       0x57, 0xbb, 0x55, 0xc8, 0xa8, 0x50, 0x93, 0xdc, 0xbf, 0xfc,
	       0xde, 0x46, 0x8a, 0x53, 0x9f, 0x12, 0xaa, 0x7c, 0xf1, 0xdd,
	       0x89, 0x9e, 0x02, 0x27, 0x9c, 0x1a, 0xa0, 0x94, 0xf5, 0xec,
	       0x06, 0xa3, 0xdb, 0xf3, 0x3f, 0x6d, 0xfd, 0x30, 0x6d, 0xab,
	       0xcb, 0xc3, 0x72, 0xa9, 0x25, 0x35, 0x69, 0x67, 0x07, 0xaf,
	       0x9c, 0x91, 0x3a, 0x24, 0x03, 0x74, 0x59, 0xfd, 0x69, 0xa6,
	       0xfe, 0x23, 0xa4, 0x6c, 0x2f, 0xbe, 0x44, 0x56, 0x47, 0xee,
	       0xdb, 0x07, 0xc3, 0x72, 0x3f, 0x14, 0xdc, 0x16, 0xb9, 0x66,
	       0x48, 0x7c, 0x6e, 0x69, 0x6f, 0xa1, 0x05, 0xc6, 0x36, 0x08,
	       0x01, 0xdd, 0x1c, 0xb8, 0x52, 0xf4, 0x86, 0x96, 0x85, 0x39,
	       0x89, 0xb0, 0x31, 0x67, 0x62, 0xc5, 0x52, 0x91, 0x72, 0xd7,
	       0x96, 0x8c, 0xe1, 0x0a, 0x02, 0x6a, 0xfe, 0x82, 0xca, 0xc0,
	       0x34, 0xc9, 0xbc, 0x45, 0xa7, 0xc0, 0x4b, 0xa0, 0x7c, 0x7c,
	       0xcc, 0x29, 0xe5, 0x8c, 0xf6, 0x91, 0x65, 0x33, 0xf1, 0x7b,
	       0xda, 0x55, 0x69, 0x93, 0x2d, 0x4e, 0xb9, 0xb4, 0x7f, 0x56,
	       0xe6, 0x80, 0xbe, 0x23, 0x4a, 0x4a, 0x65, 0xa6, 0xab, 0xa2,
	       0x40, 0xb1, 0x75, 0x62, 0x13, 0xc1, 0xfd, 0x52, 0xe1, 0xbb,
	       0x7b, 0xb1, 0x7f, 0x8a, 0x0c, 0x27, 0x35, 0xec, 0x27, 0x3b,
	       0xa5, 0xe7, 0x75, 0xb8, 0xe3, 0xc4, 0xcf, 0x4d, 0x8a, 0x02,
	       0x57, 0x57, 0x16, 0xa2, 0x8e, 0x9d, 0x87, 0x5a, 0x32, 0xb6,
	       0xf6, 0x1d, 0xf5, 0xe3, 0xd7, 0xcf, 0x79, 0xc8, 0x77, 0x74,
	       0xdc, 0xe5, 0xba, 0xde, 0x5c, 0x22, 0xad, 0xc0, 0xfa, 0x67,
	       0xf3, 0x26, 0xbf, 0xcc, 0xd4, 0x88, 0xd5, 0xda, 0x87, 0x4d,
	       0x9d, 0x99, 0xc1, 0xce, 0xa4, 0x9a, 0xda, 0x99, 0xa5, 0xa2,
	       0xe1, 0xc5, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x81, 0xaa,
	       0x30, 0x81, 0xa7, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23,
	       0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xc3, 0x9c, 0xf3, 0xfc,
	       0xd3, 0x46, 0x08, 0x34, 0xbb, 0xce, 0x46, 0x7f, 0xa0, 0x7c,
	       0x5b, 0xf3, 0xe2, 0x08, 0xcb, 0x59, 0x30, 0x0f, 0x06, 0x09,
	       0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x01, 0x05, 0x04,
	       0x02, 0x05, 0x00, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e,
	       0x04, 0x16, 0x04, 0x14, 0xfa, 0x58, 0xdb, 0x09, 0x53, 0xbc,
	       0x19, 0xc5, 0xe7, 0xb5, 0x8b, 0xf6, 0x10, 0xf8, 0x1e, 0x84,
	       0x6d, 0x3a, 0x8f, 0xd8, 0x30, 0x13, 0x06, 0x03, 0x55, 0x1d,
	       0x25, 0x04, 0x0c, 0x30, 0x0a, 0x06, 0x08, 0x2b, 0x06, 0x01,
	       0x05, 0x05, 0x07, 0x03, 0x09, 0x30, 0x0c, 0x06, 0x03, 0x55,
	       0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30,
	       0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04,
	       0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x21, 0x06, 0x03, 0x55,
	       0x1d, 0x11, 0x04, 0x1a, 0x30, 0x18, 0xa4, 0x16, 0x30, 0x14,
	       0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13,
	       0x09, 0x54, 0x47, 0x56, 0x2d, 0x42, 0x2d, 0x32, 0x31, 0x34,
	       0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
	       0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00,
	       0x3e, 0x45, 0xce, 0x3d, 0x53, 0x8c, 0x88, 0xcd, 0xde, 0xf1,
	       0x38, 0x0c, 0x00, 0x7a, 0x7e, 0x22, 0xe7, 0x1a, 0xa5, 0xbe,
	       0xee, 0x1c, 0x17, 0x20, 0xc3, 0x65, 0x68, 0x86, 0x27, 0x83,
	       0x62, 0xd7, 0xdc, 0x1d, 0x6c, 0xfa, 0x24, 0x2e, 0x66, 0x50,
	       0xe5, 0xe0, 0x42, 0xa5, 0x73, 0x67, 0x2a, 0xea, 0x5a, 0x17,
	       0x20, 0x3b, 0x14, 0xd4, 0x74, 0x14, 0xbd, 0x18, 0x60, 0xbe,
	       0xa6, 0x46, 0xb1, 0xc2, 0x82, 0xc9, 0xb6, 0x99, 0x67, 0x56,
	       0xbe, 0x17, 0xda, 0x78, 0x05, 0x48, 0x65, 0x9d, 0x48, 0xb5,
	       0xda, 0x1d, 0x52, 0x59, 0x2a, 0xac, 0x09, 0x2d, 0x29, 0x18,
	       0x96, 0xc1, 0x58, 0x79, 0xfc, 0x73, 0x0b, 0x70, 0x4d, 0x31,
	       0x61, 0x80, 0xc7, 0x77, 0x02, 0xf1, 0x12, 0xb3, 0x80, 0x6f,
	       0xb9, 0x05, 0x69, 0xcf, 0x4f, 0x80, 0x7d, 0xf5, 0x06, 0xe6,
	       0x2e, 0xc7, 0x53, 0x99, 0x8b, 0x07, 0xc7, 0x7a, 0xe6, 0xf3,
	       0x12, 0x86, 0xd1, 0xbb, 0x8a, 0x8a, 0xfb, 0x9d, 0xd1, 0x0b,
	       0xe7, 0x9f, 0x12, 0x06, 0xfb, 0x7d, 0x8e, 0xe7, 0xb7, 0x39,
	       0xe0, 0x3c, 0xd0, 0xe8, 0x35, 0x44, 0x28, 0xb7, 0xcb, 0xee,
	       0xef, 0xa7, 0x14, 0xfa, 0x0e, 0x34, 0xaf, 0x78, 0x59, 0x1e,
	       0x91, 0xd9, 0xe0, 0x9b, 0x3c, 0x9e, 0x3a, 0xbf, 0xf5, 0xf5,
	       0x11, 0x5b, 0x04, 0x48, 0xcd, 0x3a, 0x3f, 0xee, 0x46, 0x6d,
	       0x69, 0x68, 0x39, 0xc1, 0x4d, 0x54, 0xfd, 0x6c, 0x27, 0x1e,
	       0x5b, 0x58, 0x00, 0xbb, 0x4f, 0x1b, 0x12, 0xd3, 0xbb, 0x46,
	       0xf4, 0x7c, 0x4a, 0x44, 0xb5, 0xcb, 0x4f, 0xf2, 0x3d, 0xc3,
	       0x51, 0xfc, 0x7a, 0x2c, 0x59, 0xd0, 0x82, 0x73, 0xe3, 0x88,
	       0xfc, 0x25, 0x4c, 0x35, 0x6f, 0x88, 0x85, 0xff, 0xad, 0x8c,
	       0x83, 0xc4, 0x76, 0x58, 0x6b, 0xfa, 0xf2, 0xed, 0x5b, 0x95,
	       0xd9, 0x07, 0x55, 0x58, 0xfe, 0x08 ) );

/** Time at which OCSP responses are valid */
static time_t test_time = 1337062083ULL; /* Tue 15 May 2012 06:08:03 */

/** Time at which OCSP responses are not valid */
static time_t test_stale = 1375573111ULL; /* Sat Aug  3 23:38:31 2013 */

/** Time at which "vultr" OCSP response (generated more recently) is valid */
static time_t test_vultr = 1416697736ULL; /* Sat 22 Nov 23:08:56 2014 */

/**
 * Report certificate parsing test result
 *
 * @v crt		Test certificate
 */
#define ocsp_certificate_ok( crt ) do {					\
	ok ( x509_certificate ( (crt)->data, (crt)->len,		\
				&(crt)->cert ) == 0 );			\
	} while ( 0 )

/**
 * Report OCSP check creation test result
 *
 * @v test		OCSP test
 */
#define ocsp_check_ok( test ) do {					\
	ocsp_prepare_test ( (test) );					\
	ok ( ocsp_check ( (test)->cert->cert, (test)->issuer->cert,	\
			  &(test)->ocsp ) == 0 );			\
	} while ( 0 )

/**
 * Report OCSP request construction test result
 *
 * @v test		OCSP test
 */
#define ocsp_request_ok( test ) do {					\
	DBGC ( (test), "OCSPTEST %p expected request:\n", (test) );	\
	DBGC_HDA ( (test), 0, (test)->request, (test)->request_len );	\
	ok ( (test)->ocsp->request.builder.len == (test)->request_len );\
	ok ( memcmp ( (test)->ocsp->request.builder.data,		\
		      (test)->request, (test)->request_len ) == 0 );	\
	DBGC ( (test), "OCSPTEST %p generated request:\n", (test) );	\
	DBGC_HDA ( (test), 0, (test)->ocsp->request.builder.data,	\
		   (test)->ocsp->request.builder.len );			\
	} while ( 0 )

/**
 * Report OCSP response test result
 *
 * @v test		OCSP test
 */
#define ocsp_response_ok( test ) do {					\
	ok ( ocsp_response ( (test)->ocsp, (test)->response,		\
			     (test)->response_len ) == 0 );		\
	} while ( 0 )

/**
 * Report OCSP response failure test result
 *
 * @v test		OCSP test
 */
#define ocsp_response_fail_ok( test ) do {				\
	ok ( ocsp_response ( (test)->ocsp, (test)->response,		\
			     (test)->response_len ) != 0 );		\
	} while ( 0 )

/**
 * Report OCSP validation test result
 *
 * @v test		OCSP test
 * @v time		Test time
 */
#define ocsp_validate_ok( test, time ) do {				\
	ocsp_prepare_test ( (test) );					\
	ok ( ocsp_validate ( (test)->ocsp, time ) == 0 );		\
	} while ( 0 )

/**
 * Report OCSP validation failure test result
 *
 * @v test		OCSP test
 * @v time		Test time
 */
#define ocsp_validate_fail_ok( test, time ) do {			\
	ocsp_prepare_test ( (test) );					\
	ok ( ocsp_validate ( (test)->ocsp, time ) != 0 );		\
	} while ( 0 )

/**
 * Perform OCSP self-tests
 *
 */
static void ocsp_test_exec ( void ) {

	/* Parse certificates */
	ocsp_certificate_ok ( &barclays_crt );
	ocsp_certificate_ok ( &google_crt );
	ocsp_certificate_ok ( &verisign_crt );
	ocsp_certificate_ok ( &thawte_crt );
	ocsp_certificate_ok ( &startssl_crt );
	ocsp_certificate_ok ( &rapidssl_crt );
	ocsp_certificate_ok ( &vultr_crt );

	/* Parse OCSP checks */
	ocsp_check_ok ( &barclays_ocsp );
	ocsp_check_ok ( &google_ocsp );
	ocsp_check_ok ( &unauthorized_ocsp );
	ocsp_check_ok ( &unknown_ocsp );
	ocsp_check_ok ( &vultr_ocsp );

	/* "barclays" test */
	ocsp_request_ok ( &barclays_ocsp );
	ocsp_response_ok ( &barclays_ocsp );
	ocsp_validate_ok ( &barclays_ocsp, test_time );
	ocsp_validate_fail_ok ( &barclays_ocsp, test_stale );

	/* "google" test */
	ocsp_request_ok ( &google_ocsp );
	ocsp_response_ok ( &google_ocsp );
	ocsp_validate_ok ( &google_ocsp, test_time );
	ocsp_validate_fail_ok ( &google_ocsp, test_stale );

	/* "unauthorized" test */
	ocsp_request_ok ( &unauthorized_ocsp );
	ocsp_response_fail_ok ( &unauthorized_ocsp );

	/* "unknown" test */
	ocsp_request_ok ( &unknown_ocsp );
	ocsp_response_fail_ok ( &unknown_ocsp );

	/* "vultr" test */
	ocsp_request_ok ( &vultr_ocsp );
	ocsp_response_ok ( &vultr_ocsp );
	ocsp_validate_ok ( &vultr_ocsp, test_vultr );
	ocsp_validate_fail_ok ( &vultr_ocsp, test_stale );

	/* Drop OCSP check references */
	ocsp_put ( unknown_ocsp.ocsp );
	ocsp_put ( unauthorized_ocsp.ocsp );
	ocsp_put ( google_ocsp.ocsp );
	ocsp_put ( barclays_ocsp.ocsp );
	ocsp_put ( vultr_ocsp.ocsp );

	/* Drop certificate references */
	x509_put ( vultr_crt.cert );
	x509_put ( rapidssl_crt.cert );
	x509_put ( startssl_crt.cert );
	x509_put ( thawte_crt.cert );
	x509_put ( verisign_crt.cert );
	x509_put ( google_crt.cert );
	x509_put ( barclays_crt.cert );
}

/** OCSP self-test */
struct self_test ocsp_test __self_test = {
	.name = "ocsp",
	.exec = ocsp_test_exec,
};

/* Drag in algorithms required for tests */
REQUIRING_SYMBOL ( ocsp_test );
REQUIRE_OBJECT ( rsa );
REQUIRE_OBJECT ( sha1 );
