/*
 * Copyright (c) 2019 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __ARM_FEATURE_SVE

#include <algorithm>

#include "arm_gemm.hpp"


#include "../../asmlib.hpp"
#include "../../utils.hpp"

namespace arm_gemm {

void sve_smallK_hybrid_fp32_mla_1VLx8(const float *A, int lda, const float *B, float *C, int ldc, int M, int N, int K, const float *bias, Activation act, bool) {
    const long loops_count = iceildiv(N, (int)get_vector_length<float>()) - 1;
    const long ldab = lda * sizeof(float);
    const long ldcb = ldc * sizeof(float);
    const long odd_depth  = (K % 4) ? (K % 4) : 4;
    const long last_width = N - (loops_count * get_vector_length<float>());
    float nullbias[64];
    if (!bias) {
        memset(nullbias, 0, (1 * get_vector_length<float>() * sizeof(float)));
    }
    float minval = - static_cast<float>(std::numeric_limits<float>::infinity());
    float maxval =   static_cast<float>(std::numeric_limits<float>::infinity());
    const float * const minptr = &minval;
    const float * const maxptr = &maxval;

    switch(act.type)
    {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            maxval = static_cast<float>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            minval = 0.0f;
            break;
    }

    for (int y0=0; y0<M; y0+=8) {
        long loops = loops_count;
        long oob_rows = std::max(8 - (M-y0), 0);
        long temp = 0;
        const float *b_ptr0 = B;
        const float *biasptr = bias ? bias : nullbias;
        const uint64_t biasinc = bias ? get_vector_length<float>() * 1*sizeof(float) : 0;
        const float *a_ptr0 = A + (y0 * lda);

        float *c_ptr0 = C + (y0 * ldc);

        switch(K) {
            case 1:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #1\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7]\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "mov z26.d, z24.d\n"
                    "mov z27.d, z24.d\n"
                    "mov z28.d, z24.d\n"
                    "mov z29.d, z24.d\n"
                    "mov z30.d, z24.d\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #1\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z25.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "mov z26.d, z24.d\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "mov z27.d, z24.d\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "mov z29.d, z24.d\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #1\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z25.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "mov z26.d, z24.d\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "mov z27.d, z24.d\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "mov z29.d, z24.d\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "mov z25.d, z24.d\n"
                    "mov z26.d, z24.d\n"
                    "mov z27.d, z24.d\n"
                    "mov z28.d, z24.d\n"
                    "mov z29.d, z24.d\n"
                    "mov z30.d, z24.d\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 2:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #2\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "mov z26.d, z24.d\n"
                    "mov z27.d, z24.d\n"
                    "mov z28.d, z24.d\n"
                    "mov z29.d, z24.d\n"
                    "mov z30.d, z24.d\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "addvl %[b_ptr0], %[b_ptr0], #2\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z25.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "mov z26.d, z24.d\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z27.d, z24.d\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "mov z28.d, z24.d\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "mov z30.d, z24.d\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #2\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z25.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "mov z26.d, z24.d\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "mov z27.d, z24.d\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "mov z29.d, z24.d\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "mov z25.d, z24.d\n"
                    "mov z26.d, z24.d\n"
                    "mov z27.d, z24.d\n"
                    "mov z28.d, z24.d\n"
                    "mov z29.d, z24.d\n"
                    "mov z30.d, z24.d\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 3:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #3\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "mov z26.d, z24.d\n"
                    "mov z27.d, z24.d\n"
                    "mov z28.d, z24.d\n"
                    "mov z29.d, z24.d\n"
                    "mov z30.d, z24.d\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "addvl %[b_ptr0], %[b_ptr0], #3\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z25.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "mov z26.d, z24.d\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z27.d, z24.d\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "mov z28.d, z24.d\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "mov z30.d, z24.d\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "addvl %[b_ptr0], %[b_ptr0], #3\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z25.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "mov z26.d, z24.d\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z27.d, z24.d\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "mov z28.d, z24.d\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "mov z30.d, z24.d\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "mov z25.d, z24.d\n"
                    "mov z26.d, z24.d\n"
                    "mov z27.d, z24.d\n"
                    "mov z28.d, z24.d\n"
                    "mov z29.d, z24.d\n"
                    "mov z30.d, z24.d\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 4:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #4\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "mov z26.d, z24.d\n"
                    "mov z27.d, z24.d\n"
                    "mov z28.d, z24.d\n"
                    "mov z29.d, z24.d\n"
                    "mov z30.d, z24.d\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "addvl %[b_ptr0], %[b_ptr0], #4\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z25.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "mov z26.d, z24.d\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "mov z27.d, z24.d\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "mov z29.d, z24.d\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "addvl %[b_ptr0], %[b_ptr0], #4\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z25.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "mov z26.d, z24.d\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z27.d, z24.d\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "mov z28.d, z24.d\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "mov z30.d, z24.d\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "mov z25.d, z24.d\n"
                    "mov z26.d, z24.d\n"
                    "mov z27.d, z24.d\n"
                    "mov z28.d, z24.d\n"
                    "mov z29.d, z24.d\n"
                    "mov z30.d, z24.d\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 5:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #5\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z31.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z27.d, z24.d\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #5\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "mov z27.d, z24.d\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #5\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 6:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #6\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z31.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z27.d, z24.d\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #6\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z27.d, z24.d\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #6\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 7:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #7\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z31.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #7\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z27.d, z24.d\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #7\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 8:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z31.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 9:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z31.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #1\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #1\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #1\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #1\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 10:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z31.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #2\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #2\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #2\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #2\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 11:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z31.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #3\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #3\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #3\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #3\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 12:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z31.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #4\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #4\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #4\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #4\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 13:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z31.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #5\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #5\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #5\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #5\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 14:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z31.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #6\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #6\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #6\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #6\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 15:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z31.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #7\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #7\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #7\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #7\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 16:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z31.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 17:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z31.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #1\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #1\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #1\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #1\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 18:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z31.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #2\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #2\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #2\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #2\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 19:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z31.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #3\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #3\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #3\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #3\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 20:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z31.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #4\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #4\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #4\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #4\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 21:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z31.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #5\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x50]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x50]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x50]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x50]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x50]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x50]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x50]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x50]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #5\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x50]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x50]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x50]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x50]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x50]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x50]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x50]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x50]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #5\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x50]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x50]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x50]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x50]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x50]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x50]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x50]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x50]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #5\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x50]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x50]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x50]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x50]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x50]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x50]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x50]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x50]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 22:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z31.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #6\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x50]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x50]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x50]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x50]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x50]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x50]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x50]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x50]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #6\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x50]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x50]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x50]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x50]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x50]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x50]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x50]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x50]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #6\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x50]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x50]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x50]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x50]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x50]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x50]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x50]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x50]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #6\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x50]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x50]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x50]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x50]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x50]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x50]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x50]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x50]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 23:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z31.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #7\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x50]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x50]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x50]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x50]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x50]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x50]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x50]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x50]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #7\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x50]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x50]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x50]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x50]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x50]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x50]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x50]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x50]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #7\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x50]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x50]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x50]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x50]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x50]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x50]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x50]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x50]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x40]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #7\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x50]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x50]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x50]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x50]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x50]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x50]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x50]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x50]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            default:
            case 24:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ptrue p7.s\n"
                    "whilelt p6.s, %[temp], %[odd_depth]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "cbz %[loops], 2f\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z31.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x40]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x50]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x50]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x50]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x50]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x50]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x50]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x50]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x50]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p7/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x40]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x50]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x50]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x50]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x50]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x50]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x50]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x50]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x50]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "mov z25.d, z24.d\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x40]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x50]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x50]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x50]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x50]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x50]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x50]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x50]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x50]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "b 5f\n"
                    "2:\n"
                    "ld1w z24.s, p0/z, [%[biasptr]]\n"
                    "add %[biasptr], %[biasptr], %[biasinc]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0]]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1]\n"
                    "mov z25.d, z24.d\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2]\n"
                    "mov z26.d, z24.d\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3]\n"
                    "mov z27.d, z24.d\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4]\n"
                    "mov z28.d, z24.d\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5]\n"
                    "mov z29.d, z24.d\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6]\n"
                    "mov z30.d, z24.d\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7]\n"
                    "mov z31.d, z24.d\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x10]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x10]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x10]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x10]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x10]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x10]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x10]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x10]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x20]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x20]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x20]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x20]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x20]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x20]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x20]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x20]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "ld1w z16.s, p7/z, [%[b_ptr0]]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "ld1w z17.s, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "ld1w z18.s, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x30]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x30]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x30]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x30]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x30]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x30]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x30]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x30]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "ld1w z19.s, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "ld1w z20.s, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "ld1w z21.s, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "ld1w z22.s, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "ld1rqw z0.s, p7/z, [%[a_ptr0], #0x40]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "ld1rqw z1.s, p7/z, [a_ptr1, #0x40]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "ld1rqw z2.s, p7/z, [a_ptr2, #0x40]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "ld1rqw z3.s, p7/z, [a_ptr3, #0x40]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "ld1rqw z4.s, p7/z, [a_ptr4, #0x40]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "ld1rqw z5.s, p7/z, [a_ptr5, #0x40]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "ld1rqw z6.s, p7/z, [a_ptr6, #0x40]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "ld1w z23.s, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "fmla z24.s, z16.s, z0.s[0]\n"
                    "ld1rqw z7.s, p7/z, [a_ptr7, #0x40]\n"
                    "fmla z25.s, z16.s, z1.s[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "fmla z26.s, z16.s, z2.s[0]\n"
                    "fmla z27.s, z16.s, z3.s[0]\n"
                    "fmla z28.s, z16.s, z4.s[0]\n"
                    "fmla z29.s, z16.s, z5.s[0]\n"
                    "fmla z30.s, z16.s, z6.s[0]\n"
                    "fmla z31.s, z16.s, z7.s[0]\n"
                    "fmla z24.s, z17.s, z0.s[1]\n"
                    "fmla z25.s, z17.s, z1.s[1]\n"
                    "fmla z26.s, z17.s, z2.s[1]\n"
                    "fmla z27.s, z17.s, z3.s[1]\n"
                    "fmla z28.s, z17.s, z4.s[1]\n"
                    "fmla z29.s, z17.s, z5.s[1]\n"
                    "fmla z30.s, z17.s, z6.s[1]\n"
                    "fmla z31.s, z17.s, z7.s[1]\n"
                    "fmla z24.s, z18.s, z0.s[2]\n"
                    "fmla z25.s, z18.s, z1.s[2]\n"
                    "fmla z26.s, z18.s, z2.s[2]\n"
                    "fmla z27.s, z18.s, z3.s[2]\n"
                    "fmla z28.s, z18.s, z4.s[2]\n"
                    "fmla z29.s, z18.s, z5.s[2]\n"
                    "fmla z30.s, z18.s, z6.s[2]\n"
                    "fmla z31.s, z18.s, z7.s[2]\n"
                    "fmla z24.s, z19.s, z0.s[3]\n"
                    "ld1rqw z0.s, p6/z, [%[a_ptr0], #0x50]\n"
                    "fmla z25.s, z19.s, z1.s[3]\n"
                    "ld1rqw z1.s, p6/z, [a_ptr1, #0x50]\n"
                    "fmla z26.s, z19.s, z2.s[3]\n"
                    "ld1rqw z2.s, p6/z, [a_ptr2, #0x50]\n"
                    "fmla z27.s, z19.s, z3.s[3]\n"
                    "ld1rqw z3.s, p6/z, [a_ptr3, #0x50]\n"
                    "fmla z28.s, z19.s, z4.s[3]\n"
                    "ld1rqw z4.s, p6/z, [a_ptr4, #0x50]\n"
                    "fmla z29.s, z19.s, z5.s[3]\n"
                    "ld1rqw z5.s, p6/z, [a_ptr5, #0x50]\n"
                    "fmla z30.s, z19.s, z6.s[3]\n"
                    "ld1rqw z6.s, p6/z, [a_ptr6, #0x50]\n"
                    "fmla z31.s, z19.s, z7.s[3]\n"
                    "ld1rqw z7.s, p6/z, [a_ptr7, #0x50]\n"
                    "fmla z24.s, z20.s, z0.s[0]\n"
                    "fmla z25.s, z20.s, z1.s[0]\n"
                    "fmla z26.s, z20.s, z2.s[0]\n"
                    "fmla z27.s, z20.s, z3.s[0]\n"
                    "fmla z28.s, z20.s, z4.s[0]\n"
                    "fmla z29.s, z20.s, z5.s[0]\n"
                    "fmla z30.s, z20.s, z6.s[0]\n"
                    "fmla z31.s, z20.s, z7.s[0]\n"
                    "fmla z24.s, z21.s, z0.s[1]\n"
                    "fmla z25.s, z21.s, z1.s[1]\n"
                    "fmla z26.s, z21.s, z2.s[1]\n"
                    "fmla z27.s, z21.s, z3.s[1]\n"
                    "fmla z28.s, z21.s, z4.s[1]\n"
                    "fmla z29.s, z21.s, z5.s[1]\n"
                    "fmla z30.s, z21.s, z6.s[1]\n"
                    "fmla z31.s, z21.s, z7.s[1]\n"
                    "fmla z24.s, z22.s, z0.s[2]\n"
                    "fmla z25.s, z22.s, z1.s[2]\n"
                    "fmla z26.s, z22.s, z2.s[2]\n"
                    "fmla z27.s, z22.s, z3.s[2]\n"
                    "fmla z28.s, z22.s, z4.s[2]\n"
                    "fmla z29.s, z22.s, z5.s[2]\n"
                    "fmla z30.s, z22.s, z6.s[2]\n"
                    "fmla z31.s, z22.s, z7.s[2]\n"
                    "fmla z24.s, z23.s, z0.s[3]\n"
                    "fmla z25.s, z23.s, z1.s[3]\n"
                    "fmla z26.s, z23.s, z2.s[3]\n"
                    "fmla z27.s, z23.s, z3.s[3]\n"
                    "fmla z28.s, z23.s, z4.s[3]\n"
                    "fmla z29.s, z23.s, z5.s[3]\n"
                    "fmla z30.s, z23.s, z6.s[3]\n"
                    "fmla z31.s, z23.s, z7.s[3]\n"
                    "5:\n"
                    "ld1rw z22.s, p7/z, [%[minptr]]\n"
                    "ld1rw z23.s, p7/z, [%[maxptr]]\n"
                    "fmax z24.s, p7/m, z24.s, z22.s\n"
                    "fmax z25.s, p7/m, z25.s, z22.s\n"
                    "fmax z26.s, p7/m, z26.s, z22.s\n"
                    "fmax z27.s, p7/m, z27.s, z22.s\n"
                    "fmin z24.s, p7/m, z24.s, z23.s\n"
                    "fmin z25.s, p7/m, z25.s, z23.s\n"
                    "fmin z26.s, p7/m, z26.s, z23.s\n"
                    "fmin z27.s, p7/m, z27.s, z23.s\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "fmax z28.s, p7/m, z28.s, z22.s\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "fmax z29.s, p7/m, z29.s, z22.s\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "fmax z30.s, p7/m, z30.s, z22.s\n"
                    "fmin z28.s, p7/m, z28.s, z23.s\n"
                    "fmax z31.s, p7/m, z31.s, z22.s\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "fmin z29.s, p7/m, z29.s, z23.s\n"
                    "fmin z30.s, p7/m, z30.s, z23.s\n"
                    "fmin z31.s, p7/m, z31.s, z23.s\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [biasptr] "+r" (biasptr)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width), [biasinc] "r" (biasinc), [minptr] "r" (minptr), [maxptr] "r" (maxptr)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
        }
    }
}

} // namespace arm_gemm

#endif // __ARM_FEATURE_SVE
